package lookup;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Collection;
import java.util.Collections;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.Vector;

import lookup.Rule.Position;

public class LookUpDictionary {

	/**
	 * 
	 */
	private static final long serialVersionUID = 5689122285204299967L;

	private TreeMap<Character, Node> words = new TreeMap<Character, Node>();
	
	private TreeMap<Character, Node> variants = new TreeMap<Character, Node>();

	private TreeSet<Rule> rules = new TreeSet<Rule>();

//for soundex lookup
	private TreeMap<Character,Node> soundex = new TreeMap<Character, Node>();

	private transient InputStream realListIS, ruleListIS, variantListIS;
	
	private transient static globals.Globals global = globals.Globals.getInstance();
	
	private transient boolean wordsChanged, variantsChanged, rulesChanged;
	private transient String wordsLog = "", variantsLog = "", rulesLog = "";
	
	private static LookUpDictionary _instance = null;
	
	private transient Vector<DataChangeListener> dataChangeListeners = new Vector<DataChangeListener>();
	
	protected LookUpDictionary() {

	}
	
	public static LookUpDictionary getInstance() {
		if(_instance == null) {
			_instance = new LookUpDictionary();
			
			File wordsSource = new File("saved data/words.txt");
			File variantsSource = new File("saved data/variants.txt");
			File rulesSource = new File("saved data/rules.txt");
			wordsSource.getParentFile().mkdirs();
			
			boolean writeWordsFile = false;			
			
			if(wordsSource.exists()) {
				global.processingMessager.writeMessage("Loading words.txt...");
				try {
					Word.readFile(wordsSource, new Word.WordAction() {
						public void process(String word, int freq, boolean userAdded) {
							_instance.addWord(word, freq, userAdded);
						}
					});
				} catch (IOException e) {
					global.exceptionHandler.showException("Error reading words.txt", e);
				}
			}
			else {
				global.processingMessager.writeMessage("Creating words.txt...");
				try {
					_instance.readInRealWordList();
					writeWordsFile = true;
				}
				catch (IOException e) {
					global.exceptionHandler.showException("Error reading pre-defined word list, please reinstall if error persists", e);
				}
			}
			
			if(variantsSource.exists()) {
				global.processingMessager.writeMessage("Loading variants.txt...");
				try {
					Variant.readFile(variantsSource, new Variant.VariantAction() {
						public void process(String variant, String replacement, boolean userAdded) {
							Word word = _instance.findWord(replacement);
							if(word == null)
								word = new Word(replacement, null, false, true, 1);
							_instance.addVariant(variant, word, userAdded);
						}
					});
				} catch (IOException e) {
					global.exceptionHandler.showException("Error reading variants.txt", e);
				}
			}
			else {
				global.processingMessager.writeMessage("Creating variants.txt...");
				try {
					_instance.readInVariantList();
					_instance.writeVariantsToFile();
					writeWordsFile = true;
				}
				catch (IOException e) {
					global.exceptionHandler.showException("Error reading pre-defined variant list, please reinstall if error persists", e);
				}
			}
			
			if(writeWordsFile) {
				try {
					_instance.writeWordsToFile();
				} catch (IOException ex) {
					global.exceptionHandler.showException("Error writing words.txt", ex);
				}
			}
			
			if(rulesSource.exists()) {
				global.processingMessager.writeMessage("Loading rules.txt...");
				try {
				Rule.readFile(rulesSource, new Rule.RuleAction() {
					public void process(String original, String replacement, Position position, boolean userAdded) {
						_instance.rules.add(new Rule(original, replacement, position, userAdded));
					}
				});
				}
				catch(IOException ex) {
					global.exceptionHandler.showException("Error reading rules.txt", ex);
				}
			}
			else {
				global.processingMessager.writeMessage("Creating rules.txt...");
				try {
					_instance.readInRules();
				}
				catch (IOException ex) {
					global.exceptionHandler.showException("Error reading pre-defined rules list, please reinstall if error persists", ex);
				}
				try {
					_instance.writeRulesToFile();
				}
				catch(IOException ex) {
					global.exceptionHandler.showException("Error writing rules.txt", ex);
				}
			}
			
			global.processingMessager.finishMessages();
		}
		return _instance;
	}
	
	public void addDataChangeListener(DataChangeListener listener) {
		dataChangeListeners.add(listener);
	}
	
	public void removeDataChangeListener(DataChangeListener listener) {
		dataChangeListeners.remove(listener);
	}
	
	private void readInVariantList() throws FileNotFoundException, IOException {
		variants = new TreeMap<Character, Node>();
		variantListIS = LookUpDictionary.class.getResourceAsStream("predef/variantlist.txt");

		if(variantListIS == null)
			throw new FileNotFoundException("Variant list not found");
		LineNumberReader textIn = new LineNumberReader(new InputStreamReader(variantListIS));
		String currentLine = "file not ready";
		String[] currentLineArray = new String[3];

		//File lFile = new File("logs/variantslog.txt");
		//lFile.getParentFile().mkdirs();
		//PrintWriter logWriter = new PrintWriter(new FileWriter(lFile));

		boolean invalidWord = false;
		
		//int wordCount = 0;
		//int variantCount = 0;

		while(!currentLine.equals("end")) {
			if(textIn.ready()) {
				currentLine = textIn.readLine();
				if(!currentLine.equals("end")) {
					currentLineArray = currentLine.split("\t");
					String currentVariant = currentLineArray[1].trim();
					String currentWord = currentLineArray[2].trim();
					
					invalidWord = false;
					if(!WordUtilities.isValidWord(currentWord) || currentWord.endsWith("\'") || currentWord.startsWith("\'")) {
						//logWriter.println(textIn.getLineNumber() + "- " + currentWord + " is invalid real word");
						invalidWord = true;
					}
					if(!WordUtilities.isValidWord(currentVariant)) {
						//logWriter.println(textIn.getLineNumber() + "- " + currentVariant + " is invalid variant");
						invalidWord = true;
					}



//add actual word to real words list
					if(!invalidWord) {
						if(addWord(currentWord, 1, false) == false) {
							//logWriter.println(textIn.getLineNumber() + "- " + currentWord + " existed in real words");
						}
						else {
							//logWriter.println(textIn.getLineNumber() + "- Added: " + currentWord + " to real words");
							//wordCount++;
						}
//add variant to variant list
						addVariant(currentVariant, findWord(currentWord), false);
						//logWriter.println(textIn.getLineNumber() + "- Added: " + currentVariant + " to variant list");
						//variantCount++;
					}
				}		
			}

			//else
				//logWriter.println(textIn.getLineNumber() + ": file not ready");
		}

		//logWriter.println(variantCount + " added to variants list sucessfully");
		//logWriter.println(wordCount + " added to real word list");
		//logWriter.close();
		//textIn.close();
		
		setVariantsChanged(true, globals.Globals.NEW_LINE + "Variants list (re)initialized");	}
	
	private void readInRealWordList() throws IOException {
		words = new TreeMap<Character, Node>();
		soundex = new TreeMap<Character, Node>();
		
		realListIS = LookUpDictionary.class.getResourceAsStream("predef/realwordlist.txt");
		
		if(realListIS == null)
			throw new FileNotFoundException("Real Word list not found");
		LineNumberReader textIn = new LineNumberReader(new InputStreamReader(realListIS));
		String currentLine = "file not ready";
		String[] currentLineArray = new String[7];
		//File lFile = new File("logs/realwordslog.txt");
		//lFile.getParentFile().mkdirs();
		//PrintWriter logWriter = new PrintWriter(new FileWriter(lFile));

		//int wordCount = 0;
		int currentRange = 0;
		int currentFreq = 0;
		
		while(!currentLine.equals("end")) {
			if(textIn.ready()) {
				currentLine = textIn.readLine();
				if(!currentLine.equals("end")) {
					currentLineArray = currentLine.split("\t");
					String currentWord;
					if(currentLineArray[3].equals(":")) {
						currentWord = currentLineArray[1].trim();
						currentFreq = Integer.parseInt(currentLineArray[4]);
						currentRange = Integer.parseInt(currentLineArray[5]);
					}
					else if(currentLineArray[1].equals("@")) {
						currentWord = currentLineArray[3].trim();
						currentFreq = Integer.parseInt(currentLineArray[4]);
					}
					else {
						currentFreq = Integer.parseInt(currentLineArray[4]);
						currentRange= Integer.parseInt(currentLineArray[5]);
						continue;
					}
					if(!WordUtilities.isValidWord(currentWord) || currentWord.endsWith("\'") || currentWord.startsWith("\'")) {
						//logWriter.println(textIn.getLineNumber() + "- " + currentWord + " not valid");
					}
					else if(currentRange < 50 || currentFreq < 1) {
						//logWriter.println(textIn.getLineNumber() + "- " + currentWord + " too low frequency");
					}
					else {
						if(addWord(currentWord, currentFreq, false) == false) {
							Word exists = findWord(currentWord);
								exists.setFreq(exists.getFreq() + currentFreq);
						}
						//else {
							//logWriter.println(textIn.getLineNumber() + "- Added: " + currentWord + " to real words");
							//wordCount++;
						//}
					}
					
				}
			}

			//else
				//logWriter.println(textIn.getLineNumber() + ": file not ready");
		}

		//logWriter.println(wordCount + " added to real word list sucessfully");
		//logWriter.close();
		textIn.close();
		
		
		
		//added 28/6/07 as 1 letter words now accepted
		addWord("I", 8875, false);
		addWord("a", 21626, false);
		
		setWordsChanged(true, globals.Globals.NEW_LINE + "Words list (re)initialized");		
	}

	private void readInRules() throws IOException {
		rules = new TreeSet<Rule>();
		ruleListIS = LookUpDictionary.class.getResourceAsStream("predef/rulelist.txt");
		
		if(ruleListIS == null)
			throw new FileNotFoundException("Rule list not found");
		LineNumberReader textIn = new LineNumberReader(new InputStreamReader(ruleListIS));
		String currentLine = "file not ready";
		String[] currentLineArray;

		//File lFile = new File("logs/ruleslog.txt");
		//lFile.getParentFile().mkdirs();
		//PrintWriter logWriter = new PrintWriter(new FileWriter(lFile));

		Position currentPos = Position.Anywhere;

		while(!currentLine.equals("end")) {
			if(textIn.ready()) {
				currentLine = textIn.readLine();
				if((!currentLine.equals("end")) && (currentLine.charAt(0) != '#')) {
					currentLineArray = currentLine.split("\t");
					if(currentLineArray[0].startsWith("^")) {
						currentPos = Position.Start;
						currentLineArray[0] = currentLineArray[0].substring(1);
					}
					else if(currentLineArray[0].endsWith("$")) {
						currentPos = Position.End;
						currentLineArray[0] = currentLineArray[0].substring(0,currentLineArray[0].length()-1);
					}
					else
						currentPos = Position.Anywhere;
					
					if(currentLineArray[1].equals(" "))
						currentLineArray[1] = "";
					Rule r = new Rule(currentLineArray[0],currentLineArray[1], currentPos, false);
					rules.add(r);
					//logWriter.println(textIn.getLineNumber() + "- Added: " + r.toString());
				}
			}

			//else
				//logWriter.println(textIn.getLineNumber() + ": file not ready");
		}
		//logWriter.println(rules.size() + " added to rules list sucessfully");
		//logWriter.close();
		//textIn.close();
		
		setRulesChanged(true, globals.Globals.NEW_LINE + "Rules list (re)initialized");
	}
	
	interface NodeProcessor {
		void processNode(Node node);
	}
	
	private void postOrder(Node node, NodeProcessor processor) {
		for(Node n : node.next.values()) {
			postOrder(n, processor);
		}
		processor.processNode(node);
	}
	
	public void writeRulesToFile() throws IOException {
		global.processingMessager.writeMessage("Writing rules.txt");
		PrintWriter writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream("saved data/rules.txt"),"UTF-8"));
		File lFile = new File("logs/rules_change_log.txt");
		lFile.getParentFile().mkdirs();
		PrintWriter logWriter = new PrintWriter(new FileWriter(lFile, true));
		try{
			for(Rule r : rules) {
				r.writeToFile(writer);
			}

			if(rulesChanged) {
				logWriter.println(rulesLog);
				setRulesChanged(false, null);
			}
		}
		finally {
			writer.close();
			logWriter.close();
			global.processingMessager.finishMessages();
		}
	}
	
	public void writeVariantsToFile() throws IOException {
		global.processingMessager.writeMessage("Writing variants.txt");
		final PrintWriter writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream("saved data/variants.txt"),"UTF-8"));
		File lFile = new File("logs/variants_change_log.txt");
		lFile.getParentFile().mkdirs();
		PrintWriter logWriter = new PrintWriter(new FileWriter(lFile, true));
		try {
			Node variantsRoot = new Node('@');
			variantsRoot.next = variants;
		
			postOrder(variantsRoot, new NodeProcessor() {
				public void processNode(Node node) {
					if(node.isWord())
						((VariantHolder) node).writeToFile(writer);
				}
			});
			
			if(variantsChanged) {
				logWriter.println(variantsLog);
				setVariantsChanged(false, null);
			}
		}
		finally {
			writer.close();
			logWriter.close();
			global.processingMessager.finishMessages();
		}
	}
	
	public void writeWordsToFile() throws IOException {
		global.processingMessager.writeMessage("Writing words.txt");
		final PrintWriter writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream("saved data/words.txt"),"UTF-8"));
		File lFile = new File("logs/words_change_log.txt");
		lFile.getParentFile().mkdirs();
		PrintWriter logWriter = new PrintWriter(new FileWriter(lFile, true));
		try {
			Node wordsRoot = new Node('@');
			wordsRoot.next = words;
		
			postOrder(wordsRoot, new NodeProcessor() {
				public void processNode(Node node) {
					if(node.isWord())
						((Word) node).writeToFile(writer);
				}
			});
			
			if(wordsChanged) {
				logWriter.println(wordsLog);
				setWordsChanged(false, null);
			}
			
		}
		finally {
			writer.close();
			logWriter.close();
			global.processingMessager.finishMessages();
		}
	}
	
	private Word findWord(String w) {
		String word = w.toLowerCase();
		char[] letters = word.toCharArray();
		TreeMap<Character, Node> current = words;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			return null;
		}
		
		//System.out.println(currentNode.key);
		
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				return null;
			}
			//for(int k=0;k<i;k++) {
			//	System.out.print("#");
			//}
			//System.out.println(currentNode.key);
		}
		if(currentNode.isWord()) {
			//System.out.println("~WORD: " + ((Word) currentNode).getWord());
			return (Word) currentNode;
		}
		
		else
			return null;
	}
	
	
	public Word addUserWord(String w) {
		Word toReturn;
		if((toReturn = findWord(w)) != null)
			return toReturn;
		
		
		String word = w.toLowerCase();
		char[] letters = word.toCharArray();
		TreeMap<Character, Node> current = words;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			current.put(letters[0], currentNode = new Node(letters[0]));
		}
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				current.put(letters[i], currentNode = new Node(letters[i]));
			}
		}
		if(currentNode.isWord()) {
			return (Word) currentNode;
		}
		else {
			char lastletter = letters[letters.length-1];
			current.remove(lastletter);
			current.put(lastletter, toReturn = new Word(word, currentNode, true, true, 1));
			//userWords.add(word);
			addSoundex(toReturn);
			return toReturn;
		}
	}
	
	public boolean removeVariant(String v) {
		
		VariantHolder foundVariant;
		
		Vector<Node> nodePath = new Vector<Node>();
		
		String variant = v.toLowerCase();
		char[] letters = variant.toCharArray();
		TreeMap<Character, Node> current = variants;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			return false;
		}
				
		for(int i=1; i<letters.length; i++) {
			nodePath.add(0, currentNode);
			current = currentNode.next;
			currentNode = current.get(letters[i]);
			if(currentNode == null)
				return false;
		}
		if(currentNode.isWord()) {
			foundVariant = (VariantHolder) currentNode;
		}
		else
			return false;
		
		//making the removed word a node (not a word)
		nodePath.firstElement().next.put(foundVariant.key, new Node(foundVariant));
		
		//setting the current child to the newly created node above
		Node child = nodePath.firstElement().next.get(foundVariant.key);
		
		if(child.next.isEmpty()) {
			//go through each node from the node which holds the removed word in it's next hash field back to root node deleting if the next is now empty
			for(Node node : nodePath) {
				node.next.remove(child.key);

				if(node.next.isEmpty()) {
					child = node;
				}
				else return true;
			}
		}

		return true;
	}
	
	public boolean removeWord(String w) {
		
		Word foundWord;
		
		Vector<Node> nodePath = new Vector<Node>();
		
		String word = w.toLowerCase();
		char[] letters = word.toCharArray();
		TreeMap<Character, Node> current = words;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			return false;
		}
				
		for(int i=1; i<letters.length; i++) {
			nodePath.add(0, currentNode);
			current = currentNode.next;
			currentNode = current.get(letters[i]);
			if(currentNode == null)
				return false;
		}
		if(currentNode.isWord()) {
			foundWord = (Word) currentNode;
		}
		else
			return false;
		
		//making the removed word a node (not a word)
		nodePath.firstElement().next.put(foundWord.key, new Node(foundWord));
		
		//setting the current child to the newly created node above
		Node child = nodePath.firstElement().next.get(foundWord.key);
		
		if(child.next.isEmpty()) {
			//go through each node from the node which holds the removed word in it's next hash field back to root node deleting if the next is now empty
			for(Node node : nodePath) {
				node.next.remove(child.key);

				if(node.next.isEmpty()) {
					child = node;
				}
				else return true;
			}
		}

		return true;
	}
	
	private boolean addWord(String w, int freq, boolean userAdded) {
		String word = w.toLowerCase();
		char[] letters = word.toCharArray();
		TreeMap<Character, Node> current = words;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			current.put(letters[0], currentNode = new Node(letters[0]));
		}
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				current.put(letters[i], currentNode = new Node(letters[i]));
			}
		}
		if(currentNode.isWord()) {
			return false;
		}
		else {
			char lastletter = letters[letters.length-1];
			current.remove(lastletter);
			Word newWord = new Word(word, currentNode, userAdded, true, freq);
			current.put(lastletter, newWord);
			addSoundex(newWord);
			return true;
		}
	}
	
	private TreeSet<Variant> findVariant(String w) { //returns empty treeset if not found
		String word = w.toLowerCase();
		char[] letters = word.toCharArray();
		TreeMap<Character, Node> current = variants;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			return new TreeSet<Variant>();
		}
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				return new TreeSet<Variant>();
			}
		}
		if(currentNode.isWord()) {
			return ((VariantHolder) currentNode).getVariants();
		}
		
		else
			return new TreeSet<Variant>();
	}
	
	private boolean addSoundex(Word w) {
		String wordsoundex = w.getSoundexCode();
		char[] letters = wordsoundex.toCharArray();
		TreeMap<Character, Node> current = soundex;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			current.put(letters[0], currentNode = new Node(letters[0]));
		}
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				current.put(letters[i], currentNode = new Node(letters[i]));
			}
		}
		if(currentNode.isWord()) {
			return ((SoundexHolder) currentNode).addWord(w);
		}
		else {
			char lastletter = letters[letters.length-1];
			current.remove(lastletter);
			SoundexHolder sh = new SoundexHolder(wordsoundex, currentNode);
			current.put(lastletter, sh);
			return sh.addWord(w);
		}
	}
	
	private TreeSet<Word> findWordWithSoundex(String s) { //returns empty treeset if not found
		char[] letters = s.toCharArray();
		TreeMap<Character, Node> current = soundex;
		Node currentNode;
		
		if(letters.length == 0) {
			return new TreeSet<Word>();
		}
		
		if((currentNode = current.get(letters[0])) == null) {
			return new TreeSet<Word>();
		}
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				return new TreeSet<Word>();
			}
		}
		if(currentNode.isWord()) {
			return ((SoundexHolder) currentNode).getWords();
		}
		
		else
			return new TreeSet<Word>();
	}
	

	private boolean addVariant(String v, Word rep, boolean userAdded) {
		String variant = v.toLowerCase();
		char[] letters = variant.toCharArray();
		TreeMap<Character, Node> current = variants;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			current.put(letters[0], currentNode = new Node(letters[0]));
		}
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				current.put(letters[i], currentNode = new Node(letters[i]));
			}
		}
		if(currentNode.isWord()) {
			return ((VariantHolder) currentNode).addVariant(rep, userAdded);
			
		}
		else {
			char lastletter = letters[letters.length-1];
			current.remove(lastletter);
			VariantHolder vh = new VariantHolder(variant, currentNode);
			current.put(lastletter, vh);
			return vh.addVariant(rep, userAdded);
		}
	}
	
	public boolean addUserVariant(String v, Word rep) { //returns true if added
		String variant = v.toLowerCase();
		char[] letters = variant.toCharArray();
		TreeMap<Character, Node> current = variants;
		Node currentNode;
		if((currentNode = current.get(letters[0])) == null) {
			current.put(letters[0], currentNode = new Node(letters[0]));
		}
		for(int i=1; i<letters.length; i++) {
			current = currentNode.next;
			if((currentNode = current.get(letters[i])) == null) {
				current.put(letters[i], currentNode = new Node(letters[i]));
			}
		}
		if(currentNode.isWord()) {
			return ((VariantHolder) currentNode).addVariant(rep, false);
		}
		else {
			char lastletter = letters[letters.length-1];
			current.remove(lastletter);
			VariantHolder vh = new VariantHolder(variant, currentNode);
			current.put(lastletter, vh);
			if(vh.addVariant(rep, true)) {
				//userVariants.add(new Variant(v, rep, true));
				return true;
			}
			else
				return false;
		}
	}


	public Word checkWords(String w) { //returns null if not found
		return findWord(w);
	}
	
	private SuggestedReplacement getHashRemovalMatch(String wtf, ConfidenceWeights rs) {
		String[] s = wtf.split("-");
		if(s.length < 2)
			return null;
		if(checkWords(s[0]) == null)
			return null;

		String newWordString = s[0];
		for(int i=1;i<s.length;i++) {
			if(checkWords(s[i]) == null)
				return null;
			else
				newWordString += (" " + s[i]);
		}

		SuggestedReplacement toReturn = new SuggestedReplacement(new Word(newWordString, null, false, true, 1), wtf, rs);
		toReturn.setLetterReplacement(true);
		toReturn.setSoundex(true);
		return toReturn;
	}

	public Vector<SuggestedReplacement> findReplacements(String wtf, String soundex, ConfidenceWeights rs) { //returns empty vector if non found

		TreeSet<Variant> variantMatches = findVariant(wtf);
		TreeSet<String> lrMatches = checkLetterReplacements(wtf);
		TreeSet<Word> soundexMatches = findWordWithSoundex(soundex);
		


		Vector<String> stringResults = new Vector<String>(100,50);

		Vector<SuggestedReplacement> results = new Vector<SuggestedReplacement>(100,50);

		SuggestedReplacement hashRem = getHashRemovalMatch(wtf, rs);
		if(hashRem != null) {
			results.add(hashRem);
			stringResults.add(hashRem.getReplacement().toString());
		}

		SuggestedReplacement toAdd;
		int index;

		for(Variant current : variantMatches) {
			if(!stringResults.contains(current.getActual().toString())) {
				toAdd = new SuggestedReplacement(current.getActual(), wtf, rs);
				toAdd.setKnownVariant(true, current.isUserAdded());
				toAdd.setSoundex(soundex);
				results.add(toAdd);
				stringResults.add(current.getActual().toString());
			}
		}

		Word realWordFound;

		for(String current : lrMatches) {
			index = stringResults.indexOf(current);
			if(index<0) {
				realWordFound = checkWords(current);
				if(realWordFound==null)
					toAdd = new SuggestedReplacement(new Word(current, null, false, false, 1), wtf, rs);
				else
					toAdd = new SuggestedReplacement(realWordFound, wtf, rs);
				toAdd.setLetterReplacement(true);
				toAdd.setSoundex(soundex);
				results.add(toAdd);
				stringResults.add(current);
			}
			else
				results.get(index).setLetterReplacement(true);
		}

		for(Word current : soundexMatches) {
			index = stringResults.indexOf(current.toString());
			if(index<0) {
				toAdd = new SuggestedReplacement(current, wtf, rs);
				toAdd.setSoundex(soundex);
				results.add(toAdd);
				stringResults.add(current.toString());
			}
			//no need to edit if already exists, as soundex has been added
		}

		//filter results
		Collections.sort(results);
		int nonDictSoundexAdded = 0;
		int nonDictNotSoundexAdded = 0;
		int dictAdded = 0;
		Vector<SuggestedReplacement> filtered = new Vector<SuggestedReplacement>(30, 10);
		for(SuggestedReplacement rep : results) {
			if(WordUtilities.isLikelyReplacement(wtf, rep.getReplacementString(), rep.getEditDistance())) {
				if(rep.isInDictionary()) {
					if(dictAdded<20) {
						filtered.add(rep);
						dictAdded++;
					}
				}
				else if(rep.isSoundex()) {
					if(nonDictSoundexAdded<6) {
						filtered.add(rep);
						nonDictSoundexAdded++;
					}
				}
				else {
					if(nonDictNotSoundexAdded<6) {
						filtered.add(rep);
						nonDictNotSoundexAdded++;
					}
				}
			}
		}
		return filtered;
	}

	private TreeSet<String> checkLetterReplacements(String wtf) { //returns vector of real words found that are candidates, if none are found, vector is empty

		Vector<FlaggedString> toCheck = new Vector<FlaggedString>(300,50);
		Vector<FlaggedString> found =  new Vector<FlaggedString>(300,50);
		Vector<FlaggedString> foundVariants = new Vector<FlaggedString>(200,50);

		toCheck.add(new FlaggedString(wtf.toLowerCase()));



		FlaggedString current;

		while(!toCheck.isEmpty()) {
			current = toCheck.firstElement();
			for(Rule rule : rules) {
				foundVariants = current.applyRule(rule);
				if(toCheck.size() < 5000) {                     // to stop the system getting too many checks to handle, this rarely happens, only for extraordinary long strings
					found.addAll(foundVariants);
					toCheck.addAll(foundVariants);
				}
			}
			toCheck.remove(current);
		}
			
		TreeSet<String> filtered = new TreeSet<String>();

		for(FlaggedString fs : found) {
			filtered.add(fs.toString());
		}

		return filtered;
	}

	public TreeSet<Rule> getRules() {
		return rules;
	}
	
	public void setRules(Collection<Rule> newRules) {
		rules = new TreeSet<Rule>();
		rules.addAll(newRules);
	}

	/**
	 * @return the rulesChanged
	 */
	public boolean isRulesChanged() {
		return rulesChanged;
	}

	/**
	 * @return the variantsChanged
	 */
	public boolean isVariantsChanged() {
		return variantsChanged;
	}

	/**
	 * @return the wordsChanged
	 */
	public boolean isWordsChanged() {
		return wordsChanged;
	}

	/**
	 * @param rulesChanged the rulesChanged to set
	 */
	public void setRulesChanged(boolean rulesChanged, String log) {
		this.rulesChanged = rulesChanged;
		if(rulesChanged)
			rulesLog += log;
		else
			rulesLog = "";
		
		for(DataChangeListener l : dataChangeListeners) {
			l.rulesChanged(rulesChanged);
		}
	}

	/**
	 * @param variantsChanged the variantsChanged to set
	 */
	public void setVariantsChanged(boolean variantsChanged, String log) {
		this.variantsChanged = variantsChanged;
		if(variantsChanged)
			variantsLog += log;
		else
			variantsLog = "";
		
		for(DataChangeListener l : dataChangeListeners) {
			l.variantsChanged(variantsChanged);
		}
	}

	/**
	 * @param wordsChanged the wordsChanged to set
	 */
	public void setWordsChanged(boolean wordsChanged, String log) {
		this.wordsChanged = wordsChanged;
		if(wordsChanged)
			wordsLog += log;
		else
			wordsLog = "";
		
		for(DataChangeListener l : dataChangeListeners) {
			l.wordsChanged(wordsChanged);
		}
	}

	/**
	 * @return the rulesLog
	 */
	public String getRulesLog() {
		return rulesLog;
	}

	/**
	 * @return the variantsLog
	 */
	public String getVariantsLog() {
		return variantsLog;
	}

	/**
	 * @return the wordsLog
	 */
	public String getWordsLog() {
		return wordsLog;
	}

	/**
	 * @param rulesLog the rulesLog to set
	 */
	public void setRulesLog(String rulesLog) {
		this.rulesLog = rulesLog;
	}

	/**
	 * @param variantsLog the variantsLog to set
	 */
	public void setVariantsLog(String variantsLog) {
		this.variantsLog = variantsLog;
	}

	/**
	 * @param wordsLog the wordsLog to set
	 */
	public void setWordsLog(String wordsLog) {
		this.wordsLog = wordsLog;
	}
	
	public static interface DataChangeListener {
		public void wordsChanged(boolean changed);
		public void variantsChanged(boolean changed);
		public void rulesChanged(boolean changed);
	}
}
