package lookup;

import java.util.Vector;

import lookup.Rule.Position;

class FlaggedString implements Cloneable {

	private String string;
	private boolean[] dirty; //each boolean represents char in string, if true then dirty, hence won't be changed again
	private int stringLength;
	private int rulesApplied;

	public FlaggedString() {
	}
	
	public FlaggedString(String s) {
		string = s;
		stringLength = string.length();
		dirty = new boolean[stringLength];
		rulesApplied = 0;
	}

	public FlaggedString(FlaggedString original) {
		string = new String(original.toString());
		stringLength = string.length();
		dirty = new boolean[stringLength];
		for(int i=0;i<stringLength;i++) {
			if(original.isDirty(i))
				dirty[i] = true;
		}
		rulesApplied = original.getRulesApplied();
		
	}

	private void setRulesApplied(int rulesApplied) {
		this.rulesApplied = rulesApplied;
	}

	private void addRuleApplied() {
		rulesApplied++;
	}
		

	private int getRulesApplied() {
		return this.rulesApplied;
	}

	public String toString() {
		return string;
	}

	private int length() {
		return stringLength;
	}

	private void setCharDirty(int i, boolean d) throws IndexOutOfBoundsException {
		if(i<0 || i>stringLength-1)
			throw new IndexOutOfBoundsException();
		else
			dirty[i] = d;
	}

	private void dirtyRange(int start, int end) throws IndexOutOfBoundsException { //from start upto but not including end
		if(end<start || start<0)
			throw new IndexOutOfBoundsException();
		else {
			for(int i=start;i<end&&i<stringLength;i++) {
				dirty[i] = true;
			}
		}
	}

	private boolean isDirty(int start, int end) throws IndexOutOfBoundsException { //from start upto but not including end
		if(end<start || start<0 || end>stringLength) {
			throw new IndexOutOfBoundsException();
		}
		else {
			for(int i=start;i<end;i++) {
				if(dirty[i])
					return true;
			}
			return false;
		}
	}

	private boolean isDirty(int i) throws IndexOutOfBoundsException {
		if(i<0 || i>stringLength-1)
			throw new IndexOutOfBoundsException();
		else
			return dirty[i];
	}				


	private boolean startsWith(String prefix) { //false if doesn't start with prefix or start is dirty
		if(prefix.length() > stringLength)
			return false;
		else if(isDirty(0,prefix.length()))
			return false;
		else
			return string.startsWith(prefix);
	}

	private boolean endsWith(String suffix) { //false if doesn't end with suffix or end is dirty
		if(suffix.length() > stringLength)
			return false;
		else if(isDirty(stringLength-suffix.length(),stringLength))
			return false;
		else
			return string.endsWith(suffix);
	}	


	private int indexOf(String toCheck) { //return -1 if substring isn't cleanly present, pos where substring starts otherwise

		int currentStart = 0;
		int toCheckLength = toCheck.length();
		if(toCheckLength > stringLength)
			return -1;
		int currentEnd = currentStart + toCheckLength;
		int result = -1;
		while(currentEnd<stringLength+1) {
			result = string.indexOf(toCheck,currentStart);
			if(result==-1)
				return -1;
			else {
				currentStart = result;
				currentEnd = currentStart + toCheckLength;
				if(isDirty(currentStart,currentEnd)) {
					currentStart++;
					currentEnd++;
				}
				else
					return currentStart;
			}
		}
		return -1;
	}

	public Vector<FlaggedString> applyRule(Rule rule) { //returns null if rule can't be applied
		Position where = rule.getWhereToApply();
		String original = rule.getOriginal();
		String replacement = rule.getReplacement();

		Vector<FlaggedString> toReturn = new Vector<FlaggedString>(10,5);

		if(stringLength < 2)
			return toReturn;

		if(rulesApplied > 2)
			return toReturn;

		switch(where) {
			case Start:
				if(startsWith(original) && !dirty[0])
					toReturn.add(replace(original,replacement,0));
				return toReturn;

			case End:
				if(endsWith(original) && !dirty[stringLength-1])
					toReturn.add(replace(original,replacement,stringLength-original.length()));
				return toReturn;
					
			case Anywhere:
				int start = indexOf(original);
				if(start>-1) {
					Vector<FlaggedString> toCheck = new Vector<FlaggedString>(20,5);
					toCheck.add(new FlaggedString(this));
					FlaggedString current, toAdd;

					while(!toCheck.isEmpty()) {
						current = toCheck.firstElement();
						start = current.indexOf(original);
						if(start==-1)
							toCheck.remove(0);
						else {
							toAdd = current.replace(original,replacement,start);
							toCheck.add(toAdd);
							toReturn.add(toAdd);
							current.dirtyRange(start,start+original.length());
							
						}
					}
				}
				return toReturn;

			default:
				return toReturn;
		}
	}

	private FlaggedString substring(int start, int end) {
		FlaggedString toReturn = new FlaggedString(string.substring(start,end));
		int trPlace = 0;  //position in new FlaggedString
		for(int i=start;i<end;i++) {
			toReturn.setCharDirty(trPlace,dirty[i]);
			trPlace++;
		}
		return toReturn;
	}

	private FlaggedString concat(FlaggedString newFS) {
		FlaggedString toReturn = new FlaggedString(string + newFS.toString());
		
		for(int i=0;i<stringLength;i++) {
			toReturn.setCharDirty(i,dirty[i]);
		}
		int toReturnLength = toReturn.length();
		int nfsPlace = 0;

		for(int i=stringLength;i<toReturnLength;i++) {
			toReturn.setCharDirty(i,newFS.isDirty(nfsPlace));
			nfsPlace++;
		}
		return toReturn;
	}

	private FlaggedString concat(String newS) {
		FlaggedString toReturn = new FlaggedString(string + newS);
		for(int i=0;i<stringLength;i++) {
			toReturn.setCharDirty(i,dirty[i]);
		}
		return toReturn;
	}

	private FlaggedString replace(String original, String replacement, int start) {
		FlaggedString before = substring(0,start);
		FlaggedString after = substring(start+original.length(),stringLength);
		FlaggedString toReturn = before.concat(replacement);
		toReturn = toReturn.concat(after);
		toReturn.dirtyRange(start,start+replacement.length());
		toReturn.setRulesApplied(rulesApplied);
		toReturn.addRuleApplied();
		return toReturn;
	}

	public boolean[] getDirty() {
		return this.dirty;
	}

	public void setDirty(boolean[] dirty) {
		this.dirty = dirty;
	}

	public String getString() {
		return this.string;
	}

	public void setString(String string) {
		this.string = string;
	}

	public int getStringLength() {
		return this.stringLength;
	}

	public void setStringLength(int stringLength) {
		this.stringLength = stringLength;
	}
}
