package lookup;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Vector;

import doc.HolderChangeListener;
import doc.InstanceHolder;
import doc.ReplacedInstance;

public class ConfidenceWeights implements HolderChangeListener<ReplacedInstance> {

	private static final long serialVersionUID = 1574820836410213387L;
	public static final int MAX_TOTAL = 100;
	public static final int MIN_TOTAL = 0;

	public static final int SUGGESTED_TOTAL = 50;

	//must add up to MAX_TOTAL:
	public double KV_SCORE = 55.0;
	public double LR_SCORE = 22.5;
	public double SE_SCORE = 22.5;	

	private static final double SCORE_MAX = 80.0;
	private static final double SCORE_MIN = 10.0;
	private static final double SCORE_CHANGE = 0.2;

	public double ED_REDUCTION = 2.0;

	private transient boolean hasChanged = false;
	private transient boolean update = true;
	
	private transient Vector<ChangeListener> changeListeners = new Vector<ChangeListener>();
	
	private static ConfidenceWeights _instance = null;
	
	private static transient globals.Globals global = globals.Globals.getInstance();
	
	protected ConfidenceWeights() {
		
	}
	
	public static ConfidenceWeights getInstance() {
		if(_instance == null) {
			_instance = new ConfidenceWeights();
			
			File conf = new File("saved data/weights.txt");
			conf.getParentFile().mkdirs();
			
			try {
				if(conf.exists()) {
					global.processingMessager.writeMessage("Loading confidence weights...");
					_instance.loadWeights(conf);
				}
				else {
					_instance.save();
				}
			} catch (FileNotFoundException e) {
				global.exceptionHandler.showException("weights.txt not found when should exist, please reinstall if error persists", e);
			} catch (IOException e) {
				global.exceptionHandler.showException("Error reading weights.txt, please reinstall if error persists", e);
			} catch (NumberFormatException ex) {
				global.exceptionHandler.showException("Weights in weights.txt must be decimal values.", ex);
			}
			finally {
				global.processingMessager.finishMessages();
			}
		}
		
		return _instance;
	}
	
	public void loadWeights(File conf) throws IOException, NumberFormatException {
		LineNumberReader textIn = new LineNumberReader(new FileReader(conf));
		
		String currentLine = "file not ready";
		String[] currentLineArray;
		
		
		try {
		
			while((currentLine = textIn.readLine()) != null) {
				currentLineArray = currentLine.split("\t");
			
				if(currentLineArray[0].contains("Known Variant:"))
					KV_SCORE = Double.parseDouble(currentLineArray[1].trim());
				else if(currentLineArray[0].contains("Letter Replacement:"))
					LR_SCORE = Double.parseDouble(currentLineArray[1].trim());
				else if(currentLineArray[0].contains("Soundex Match:"))
					SE_SCORE = Double.parseDouble(currentLineArray[1].trim());
			}
		}
		finally {
			textIn.close();
		}
	}
	
	public void addChangeListener(ChangeListener cl) {
		changeListeners.add(cl);
	}
	
	public void removeChangeListener(ChangeListener cl) {
		changeListeners.remove(cl);
	}
	

	/**
	 * @return Returns the update.
	 */
	public boolean isUpdate() {
		return update;
	}

	/**
	 * @param update The update to set.
	 */
	public void setUpdate(boolean newUpdate) {
		update = newUpdate;
	}

	public String getKnownVariantString() {
		String scoreString = Double.toString((Math.rint(KV_SCORE*10.0)/10.0));
		if(scoreString.endsWith(".0"))
			scoreString = scoreString.substring(0,scoreString.length()-2);
		return ("Known Variant (" + scoreString + "%)");
	}

	public String getLetterReplacementString() {
		String scoreString = Double.toString((Math.rint(LR_SCORE*10.0)/10.0));
		if(scoreString.endsWith(".0"))
			scoreString = scoreString.substring(0,scoreString.length()-2);
		return ("Letter Replacement (" + scoreString + "%)");
	}

	public String getSoundexString() {
		String scoreString = Double.toString((Math.rint(SE_SCORE*10.0)/10.0));
		if(scoreString.endsWith(".0"))
			scoreString = scoreString.substring(0,scoreString.length()-2);
		return ("Soundex Match (" + scoreString + "%)");
	}

	public String getEditDistanceString(SuggestedReplacement sr) {
		String scoreString = Double.toString((Math.rint(sr.getEditDistance()*ED_REDUCTION*-10.0)/10.0));
		if(scoreString.endsWith(".0"))
			scoreString = scoreString.substring(0,scoreString.length()-2);
		return ("Edit Distance is " + sr.getEditDistance() + " (" + scoreString + "%)");
	}

	public String getNotInDictString(SuggestedReplacement sr) {
		String scoreString = Double.toString((Math.rint((sr.getScore()/2)*-10.0)/10.0));
		if(scoreString.endsWith(".0"))
			scoreString = scoreString.substring(0, scoreString.length()-2);
		return ("Not in dictionary (" + scoreString + "%)");
	}

//	public static double getSuggestedScore() {
//		return SUGGESTED_TOTAL;
//	}

	private ScoreAdjustment adjustValuesForReplacementSelected(SuggestedReplacement sr) {
		
		double startKV = KV_SCORE, startLR = LR_SCORE, startSE = SE_SCORE;
		
		
		boolean kv = sr.isKnownVariant();
		boolean lr = sr.isLetterReplacement();
		boolean se = sr.isSoundex();

		int amountOfTypes = 0;
		if(kv)
			amountOfTypes++;
		if(lr)
			amountOfTypes++;
		if(se)
			amountOfTypes++;

		switch(amountOfTypes) {
			case 0:
				break;

			case 1:
				//rule out type to increase being at max

				if(kv && KV_SCORE >= SCORE_MAX)
					return new ScoreAdjustment(0.0,0.0,0.0);
				if(lr && LR_SCORE >= SCORE_MAX)
					return new ScoreAdjustment(0.0,0.0,0.0);
				if(se && SE_SCORE >= SCORE_MAX)
					return new ScoreAdjustment(0.0,0.0,0.0);


				//decrease types to decrease, but keep track of how many have been decreased
				int decreased = 0;

				if(!kv && KV_SCORE > SCORE_MIN) {
					KV_SCORE -= SCORE_CHANGE;
					decreased++;
				}
				if(!lr && LR_SCORE > SCORE_MIN) {
					LR_SCORE -= SCORE_CHANGE;
					decreased++;
				}
				if(!se && SE_SCORE > SCORE_MIN) {
					SE_SCORE -= SCORE_CHANGE;
					decreased++;
				}

				//nothing decreased, so can't increase any
				if(decreased == 0)
					return new ScoreAdjustment(0.0,0.0,0.0);

				//increase type to increase by order of how many decreased
				if(kv) {
					KV_SCORE += SCORE_CHANGE;
					if(decreased == 2)
						KV_SCORE += SCORE_CHANGE;
				}

				if(lr) {
					LR_SCORE += SCORE_CHANGE;
					if(decreased == 2)
						LR_SCORE += SCORE_CHANGE;
				}

				if(se) {
					SE_SCORE += SCORE_CHANGE;
					if(decreased == 2)
						SE_SCORE += SCORE_CHANGE;
				}
					
					KV_SCORE = Math.rint(KV_SCORE*10.0)/10.0;
					LR_SCORE = Math.rint(LR_SCORE*10.0)/10.0;
					SE_SCORE = Math.rint(SE_SCORE*10.0)/10.0;

				setHasChanged(true);
				break;

			case 2:
				//rule out type to decrease being at min

				if(!kv && KV_SCORE <= SCORE_MIN)
					return new ScoreAdjustment(0.0,0.0,0.0);
				if(!lr && LR_SCORE <= SCORE_MIN)
					return new ScoreAdjustment(0.0,0.0,0.0);
				if(!se && SE_SCORE <= SCORE_MIN)
					return new ScoreAdjustment(0.0,0.0,0.0);


				//increase types to increase, but keep track of how many have been increased
				int increased = 0;

				if(kv && KV_SCORE < SCORE_MAX) {
					KV_SCORE += SCORE_CHANGE;
					increased++;
				}
				if(lr && LR_SCORE < SCORE_MAX) {
					LR_SCORE += SCORE_CHANGE;
					increased++;
				}
				if(se && SE_SCORE < SCORE_MAX) {
					SE_SCORE += SCORE_CHANGE;
					increased++;
				}

				//nothing increased, so can't decrease any
				if(increased == 0)
					return new ScoreAdjustment(0.0,0.0,0.0);

				//decrease type to increase by order of how many decreased
				if(!kv) {
					KV_SCORE -= SCORE_CHANGE;
					if(increased == 2)
						KV_SCORE -= SCORE_CHANGE;
				}

				if(!lr) {
					LR_SCORE -= SCORE_CHANGE;
					if(increased == 2)
						LR_SCORE -= SCORE_CHANGE;
				}

				if(!se) {
					SE_SCORE -= SCORE_CHANGE;
					if(increased == 2)
						SE_SCORE -= SCORE_CHANGE;
				}
					
					KV_SCORE = Math.rint(KV_SCORE*10.0)/10.0;
					LR_SCORE = Math.rint(LR_SCORE*10.0)/10.0;
					SE_SCORE = Math.rint(SE_SCORE*10.0)/10.0;

				setHasChanged(true);
				break;

			case 3:
			default:
				return new ScoreAdjustment(0.0,0.0,0.0);
		}
		if(hasChanged)
			return new ScoreAdjustment(KV_SCORE-startKV, LR_SCORE-startLR, SE_SCORE-startSE);

		else return new ScoreAdjustment(0.0,0.0,0.0);
			
		
	}
	
	private void adjustValuesForScoreAdjustmentCancelled(ScoreAdjustment sa) {
		KV_SCORE -= sa.kvChange;
		LR_SCORE -= sa.lrChange;
		SE_SCORE -= sa.seChange;
		
	/*	if(KV_SCORE > SCORE_MAX || LR_SCORE > SCORE_MAX || SE_SCORE > SCORE_MAX || KV_SCORE < SCORE_MIN || LR_SCORE < SCORE_MIN || SE_SCORE < SCORE_MIN) {
			KV_SCORE += sa.kvChange;
			LR_SCORE += sa.lrChange;
			SE_SCORE += sa.seChange;
		}
	*/
		
		KV_SCORE = Math.rint(KV_SCORE*10.0)/10.0;
		LR_SCORE = Math.rint(LR_SCORE*10.0)/10.0;
		SE_SCORE = Math.rint(SE_SCORE*10.0)/10.0;
		
	}

	public double calculateScoreOfReplacement(SuggestedReplacement sr) {
		double score = 0.0;

		if(sr.isKnownVariant())
			score += KV_SCORE;
		if(sr.isLetterReplacement())
			score += LR_SCORE;
		if(sr.isSoundex())
			score += SE_SCORE;
		for(int i=0;i<sr.getEditDistance();i++) {
			score -= ED_REDUCTION;
		}

		score = Math.rint(score*10.0)/10.0;

		if(!sr.isInDictionary())
			score = score/2;

		return Math.min(Math.max(score, MIN_TOTAL), MAX_TOTAL);
	}

	public boolean hasChanged() {
		return hasChanged;
	}

	public void setHasChanged(boolean hasChanged) {
		this.hasChanged = hasChanged;
		
		for(ChangeListener cl : changeListeners) {
			cl.changeUpdate(hasChanged);
		}
	}

	public void save() throws IOException {
		global.processingMessager.writeMessage("Saving confidence weights...");
		PrintWriter writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream("saved data/weights.txt")));
		try {
			writer.println("Known Variant:\t" + KV_SCORE);
			writer.println("Letter Replacement:\t" + LR_SCORE);
			writer.println("Soundex Match:\t" + SE_SCORE);
			
			setHasChanged(false);
		}
		finally {
			global.processingMessager.finishMessages();
			writer.close();
		}
	}
	
	public static class ScoreAdjustment {
		double kvChange, lrChange, seChange;

		/**
		 * @param kvChange
		 * @param lrChange
		 * @param seChange
		 */
		public ScoreAdjustment(double kvChange, double lrChange, double seChange) {
			this.kvChange = kvChange;
			this.lrChange = lrChange;
			this.seChange = seChange;
		}
		
	}

	public void holderEmptied(InstanceHolder<ReplacedInstance> holder, ReplacedInstance ri) {
		if(update && holder.getConfidenceWeightsAdjustment() != null) {
			adjustValuesForScoreAdjustmentCancelled(holder.getConfidenceWeightsAdjustment());
			holder.setConfidenceWeightsAdjustment(null);
		}
	}

	public void holderFilled(InstanceHolder<ReplacedInstance> holder, ReplacedInstance ri) {
		if(update)
			holder.setConfidenceWeightsAdjustment(adjustValuesForReplacementSelected(ri.getReplacement()));
	}

	public void listChanged(InstanceHolder<ReplacedInstance> holder) {
	}

	public void instanceAboutToChange(ReplacedInstance instance) {
	}

	public void instanceChanged(ReplacedInstance instance) {
	}
	
	public static interface ChangeListener {
		public void changeUpdate(boolean changed);
	}
}
