/**
 * 
 */
package doc;

import java.util.TreeSet;
import java.util.Vector;


import javax.swing.text.BadLocationException;
import javax.swing.text.Position;
import javax.swing.text.StyledDocument;
import javax.swing.undo.AbstractUndoableEdit;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;

import lookup.ConfidenceWeights;
import lookup.SuggestedReplacement;
import lookup.Word;
import lookup.WordUtilities;

/**
 * @author barona
 *
 */
public class WordHolder implements Comparable<WordHolder> {
	
	public static final int VARIANT = 101;
	public static final int REPLACED = 102;
	public static final int CORRECT = 103;
	public static final int REAL_ERROR = 104;
	
	private StyledDocument doc;
	private String word, soundexCode;
	
	private InstanceHolder<VariantInstance> asVariant;
	private TreeSet<InstanceHolder<ReplacedInstance>> asReplaced;
	private Vector<HolderChangeListener<ReplacedInstance>> asReplacedChangeListeners;
	
	private InstanceHolder<CorrectInstance> asCorrect;
	private InstanceHolder<RealErrorInstance> asRealError;
	
	private boolean inDictionary;
	private boolean isLowFreq;
	private Word dictionaryRef;
	
	private Vector<SuggestedReplacement> replacements; //change to treeset
	private boolean areReplacementsSet;
	
	private static ConfidenceWeights confidenceWeights = ConfidenceWeights.getInstance();

	
	public WordHolder(String word, StyledDocument doc) {
		this.word = word;
		this.doc = doc;
		asVariant = new InstanceHolder<VariantInstance>(word, this);
		asReplaced = new TreeSet<InstanceHolder<ReplacedInstance>>();
		asReplacedChangeListeners = new Vector<HolderChangeListener<ReplacedInstance>>();
		
		asCorrect = new InstanceHolder<CorrectInstance>(word, this);
		asRealError = new InstanceHolder<RealErrorInstance>(word, this);
		soundexCode = WordUtilities.getSoundexCode(word);
		
		replacements = new Vector<SuggestedReplacement>();
	}
	
	public static String getTypeString(int type) {
		switch(type) {
		case VARIANT:
			return "Variant";
		case REPLACED:
			return "Replaced";
		case CORRECT:
			return "Modern Form";
		case REAL_ERROR:
			return "Uncommon";
		default:
			return "Type unrecognized";
		}
	}
	
	
	/**
	 * @return the doc
	 */
	public StyledDocument getDoc() {
		return doc;
	}

	public static class MarkFInstanceAsTEdit<F extends Instance, T extends Instance> extends AbstractUndoableEdit {
		/**
		 * 
		 */
		private static final long serialVersionUID = 1L;
		F fi;
		T ti;
		int fromType;
		int toType;
		InstanceHolder<F> from;
		InstanceHolder<T> to;
		WordHolder wh;
		boolean executed = false, undone = false;
		
		@SuppressWarnings("unchecked")
		public MarkFInstanceAsTEdit(F fi, T tempToInstance) throws InvalidInstanceChangeException { //tempToInstance is basically new T(), which is used to check type only
			super();
			this.fi = fi;
			this.ti = tempToInstance;
			wh = fi.holder;
			fromType = fi.getType();
			toType = ti.getType();
			if(fromType == toType)
				throw new InvalidInstanceChangeException("Can't mark instance as same type of instance.");
			
		
			switch(fromType) {
			case VARIANT:
				from = (InstanceHolder<F>) wh.asVariant;
				break;
				
			case REAL_ERROR:
				from = (InstanceHolder<F>) wh.asRealError;
				break;
				
			case CORRECT:
				from = (InstanceHolder<F>) wh.asCorrect;
				break;
				
			case REPLACED:
				throw new InvalidInstanceChangeException("Can't mark replaced instance as something else. Use revert back instead.");
				
			default:
				throw new InvalidInstanceChangeException("From instance type not recognised.");	
			}

			
			switch(toType) {
			case VARIANT:
				ti = (T) new VariantInstance(fi);
				to = (InstanceHolder<T>) wh.asVariant;
				break;
			
			case REAL_ERROR:
				ti = (T) new RealErrorInstance(fi, RealErrorInstance.USER_DEFINED);
				to = (InstanceHolder<T>) wh.asRealError;
				break;
				
			case CORRECT:
				ti = (T) new CorrectInstance(fi);
				to = (InstanceHolder<T>) wh.asCorrect;
				break;
				
			case REPLACED:
				throw new InvalidInstanceChangeException("Can't mark instance as replaced, use replace instead.");
				
			default:
				throw new InvalidInstanceChangeException("To instance type not recognised.");	
			}
			
			Instance p = fi;
			while((p = p.getPreviousInstance()) != null) {
				if(p.getType() == toType) {
					ti = (T) p;
					return;
				}
			}
			
			ti.setInstanceHolder(to);
		}

		public void execute() throws InvalidInstanceChangeException {
			if(!executed) {
				if(from.removeInstance(fi)) {
					if(to.addInstance(ti)) {
						executed = true;
						return;
					}
					else {
						//put fi back in from, atomic
						from.addInstance(fi);
					}
				}
			}
			else throw new InvalidInstanceChangeException("Can only execute once. " + fi + " to " + getTypeString(toType));
			throw new InvalidInstanceChangeException(fi + " could not be marked as " + getTypeString(toType));
		}

		public void undo() throws CannotUndoException {
			if(executed) {
				if(to.removeInstance(ti)) {
					if(from.addInstance(fi)) {
						undone = true;
						return;
					}
					else {
						//put ti back in to, atomic
						to.addInstance(ti);
					}
				}
			}
			throw new CannotUndoException();
		}

		public void redo() throws CannotRedoException {
			if(undone) {
				if(from.removeInstance(fi)) {
					if(to.addInstance(ti)) {
						undone = false;
						return;
					}
					else {
						//put fi back in from, atomic
						from.addInstance(fi);
					}
				}
			}
			throw new CannotRedoException();
			
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Mark instance as " + getTypeString(toType) + " (" + fi.getOriginal() + ")";
		}
		
		public String getActionName() {
			return "Mark instance as " + getTypeString(toType);
		}
		
		public int getToType() {
			return toType;
		}
		
		public T getToInstance() {
			return ti;
		}
	}
	
	public static class ReplaceFInstanceEdit<F extends VariantInstance> extends AbstractUndoableEdit {
		private static final long serialVersionUID = 1L;
		F fi;
		ReplacedInstance ri;
		boolean replaceText;
		int fromType;
		InstanceHolder<F> from;
		InstanceHolder<ReplacedInstance> to;
		boolean toIsNew = true;
		WordHolder wh;
		
		SuggestedReplacement rep;
		
		boolean rsHasChangedBefore = confidenceWeights.hasChanged();
		
		boolean executed = false, undone = false;
		
		@SuppressWarnings("unchecked")
		public ReplaceFInstanceEdit(F fi, SuggestedReplacement rep, boolean replaceText) throws InvalidInstanceChangeException {
			super();
			this.fi = fi;
			this.replaceText = replaceText;
			this.rep = rep;
			wh = fi.holder;
			fromType = fi.getType();
			
			switch(fromType) {
			case VARIANT:
				from = (InstanceHolder<F>) wh.asVariant;
				break;
				
			case REAL_ERROR:
				from = (InstanceHolder<F>) wh.asRealError;
				break;
				
			default:
				throw new InvalidInstanceChangeException("From instance type not recognised.");	
			}
			
			to = new InstanceHolder<ReplacedInstance>(wh.word, wh);
			for(HolderChangeListener<ReplacedInstance> hcl : wh.asReplacedChangeListeners) {
				to.addChangeListener(hcl);
			}
			
			for(InstanceHolder<ReplacedInstance> ih : wh.asReplaced) {
				if(!ih.isEmpty() && ih.getInstances().first().getReplacement().equals(rep)) {
					to = ih;
					toIsNew = false;

					break;	
				}
			}
			
			ri = new ReplacedInstance(fi, to, rep);			
		}
		
		public ReplacedInstance getReplacedInstance() {
			return ri;
		}

		public void execute() throws InvalidInstanceChangeException, BadLocationException {
			if(!executed) {
				ri.setupInText(replaceText);
				if(from.removeInstance(fi)) {
					if(!to.addInstance(ri)) {
						//put fi back in from, atomic
						from.addInstance(fi);
						throw new InvalidInstanceChangeException(fi + " could not be replaced");
					}
					
					if(toIsNew) {
						wh.asReplaced.add(to);
					}
					executed = true;
					return;
				}
				else {
					throw new InvalidInstanceChangeException(fi + " could not be replaced");
				}
			}
			else throw new InvalidInstanceChangeException("Can only execute once. Replace " + fi + " with " + ri.getReplacementString());
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				try {
					ri.revertInText();
				}
				catch(BadLocationException ex) {
					throw new CannotUndoException();
				}
				if(to.removeInstance(ri)) {
					if(!from.addInstance(fi)) {
						//put ti back in to, atomic
						to.addInstance(ri);
						throw new CannotUndoException();
					}
					
					if(toIsNew)
						wh.asReplaced.remove(to);
					
					confidenceWeights.setHasChanged(rsHasChangedBefore);
					
					undone = true;
					return;
				}
			}
			
			//if not returned, must have been error
			throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				try {
					ri.setupInText(replaceText);
				}
				catch(BadLocationException ex) {
					throw new CannotRedoException();
				}
				if(from.removeInstance(fi)) {
					if(!to.addInstance(ri)) {
						//put fi back in from, atomic
						from.addInstance(fi);
						throw new CannotRedoException();
					}
					
					if(toIsNew)
						wh.asReplaced.add(to);
					undone = false;
					return;
				}
			}
			
			//if not returned, must have been error
			throw new CannotRedoException();
			
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Replace (" + ri + ")";
		}
		
		public String getActionName() {
			return "Replace instance";
		}
		
		public WordHolder getWordHolder() {
			return wh;
		}
	}
	
	public static class RevertReplacedInstanceEdit extends AbstractUndoableEdit {
		private static final long serialVersionUID = 1L;
		VariantInstance vi;
		ReplacedInstance ri;
		int toType;
		InstanceHolder<ReplacedInstance> from = null;
		boolean isAlone = false;
		WordHolder wh;
		
		boolean rsHasChangedBefore = confidenceWeights.hasChanged();
		
		boolean executed = false, undone = false;
		
		public RevertReplacedInstanceEdit(ReplacedInstance ri) throws InvalidInstanceChangeException {
			super();
			this.ri = ri;
			wh = ri.holder;
			vi = ri.getPreviousInstance();
			toType = vi.getType();
			
			for(InstanceHolder<ReplacedInstance> ih : wh.asReplaced) {
				if(!ih.isEmpty() && ih.getInstances().first().getReplacement().equals(ri.getReplacement())) {
					from = ih;
					if(ih.getInstances().size() == 1)
						isAlone = true;
					break;	
				}
			}
			
			if(from == null)
				throw new InvalidInstanceChangeException("Replacement not found.");
			
		}

		public void execute() throws InvalidInstanceChangeException, BadLocationException {
			if(!executed) {
				ri.revertInText();
				if(from.removeInstance(ri)) {
					if(toType == VARIANT) {
						if(!wh.asVariant.addInstance(vi)) {
							//put fi back in from, atomic
							from.addInstance(ri);
							throw new InvalidInstanceChangeException(ri + " could not be reverted.");
						}
					}
					else if(toType == REAL_ERROR) {
						if(!wh.asRealError.addInstance((RealErrorInstance) vi)) {
							//put fi back in from, atomic
							from.addInstance(ri);
							throw new InvalidInstanceChangeException(ri + " could not be reverted.");
						}
					}
					else {
						//put fi back in from, atomic
						from.addInstance(ri);
						throw new InvalidInstanceChangeException(ri + " could not be reverted, invalid state to revert to.");
					}
							
					if(isAlone)
						wh.asReplaced.remove(from);
					executed = true;
					return;
				}
				else throw new InvalidInstanceChangeException(ri + " could not be reverted.");
			}
			else throw new InvalidInstanceChangeException("Can only execute once. Revert " + ri);
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				try {
				ri.setupInText(true);
				}
				catch(BadLocationException ex) {
					throw new CannotUndoException();
				}
				if(toType == VARIANT) {
					if(wh.asVariant.removeInstance(vi)) {
						if(!from.addInstance(ri)) {
							//put vi back in to, atomic
							wh.asVariant.addInstance(vi);
							throw new CannotUndoException();
						}
					}
					else throw new CannotUndoException();
				}
				else if(toType == REAL_ERROR) {
					if(wh.asRealError.removeInstance((RealErrorInstance) vi)) {
						if(!from.addInstance(ri)) {
							//put vi back in to, atomic
							wh.asRealError.addInstance((RealErrorInstance) vi);
							from.removeInstance(ri);
							throw new CannotUndoException();
						}
					}
					else throw new CannotUndoException();
				}
				else throw new CannotUndoException();

				if(isAlone)
					wh.asReplaced.add(from);
				undone = true;
				
				confidenceWeights.setHasChanged(rsHasChangedBefore);
				
				return;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				try {
					ri.revertInText();
				}
				catch(BadLocationException ex) {
					throw new CannotRedoException();
				}
				if(from.removeInstance(ri)) {
					if(toType == VARIANT) {
						if(!wh.asVariant.addInstance(vi)) {
							//put fi back in from, atomic
							from.addInstance(ri);
							throw new CannotRedoException();
						}
					}
					else if(toType == REAL_ERROR) {
						if(!wh.asRealError.addInstance((RealErrorInstance) vi)) {
							//put fi back in from, atomic
							from.addInstance(ri);
							throw new CannotRedoException();
						}
					}
					else {
						//put fi back in from, atomic
						from.addInstance(ri);
						throw new CannotRedoException();
					}
							
					if(isAlone)
						wh.asReplaced.remove(from);
					undone = false;
					return;
				}
				else throw new CannotRedoException();
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Revert (" + ri.revertString() + ")";
		}
		
		public String getActionName() {
			return "Revert to " + getTypeString(toType);
		}
	}
	
	public static class MarkAllFInstancesAsTEdit<F extends Instance, T extends Instance> extends AbstractUndoableEdit {

		private static final long serialVersionUID = 1L;
		TreeSet<F> fis;
		TreeSet<T> tis;
		WordHolder wh;
		int fromType, toType;
		InstanceHolder<F> from;
		InstanceHolder<T> to;
		
		boolean executed = false, undone = false;
		
		@SuppressWarnings("unchecked")
		public MarkAllFInstancesAsTEdit(WordHolder wh, F tempFrom, T tempTo) throws InvalidInstanceChangeException { //F and T are basically F() and T(), these are used to get the class type only
			super();
			this.wh = wh;
			fromType = tempFrom.getType();
			toType = tempTo.getType();
			
			switch(fromType) {
			case VARIANT:
				from = (InstanceHolder<F>) wh.asVariant;
				break;
				
			case REAL_ERROR:
				from = (InstanceHolder<F>) wh.asRealError;
				break;
				
			case CORRECT:
				from = (InstanceHolder<F>) wh.asCorrect;
				break;
				
			case REPLACED:
				throw new InvalidInstanceChangeException("Can't mark replaced instance as something else. Use revert back instead.");
				
			default:
				throw new InvalidInstanceChangeException("From instance type not recognised.");	
			}
			
			switch(toType) {
			case VARIANT:
				to = (InstanceHolder<T>) wh.asVariant;
				break;
				
			case REAL_ERROR:
				to = (InstanceHolder<T>) wh.asRealError;
				break;
				
			case CORRECT:
				to = (InstanceHolder<T>) wh.asCorrect;
				break;
				
			case REPLACED:
				throw new InvalidInstanceChangeException("Can't mark replaced instance as something else. Use revert back instead.");
				
			default:
				throw new InvalidInstanceChangeException("From instance type not recognised.");	
			}
			
			fis = new TreeSet<F>();
			tis = new TreeSet<T>();
			
			T toAdd;
			for(F fi : from.getInstances()) {
				fis.add(fi);
				
				switch(toType) {
				case VARIANT:
					toAdd = (T) new VariantInstance(fi);
					break;
				
				case REAL_ERROR:
					toAdd = (T) new RealErrorInstance(fi, RealErrorInstance.USER_DEFINED);
					break;
					
				case CORRECT:
					toAdd = (T) new CorrectInstance(fi);
					break;
				case REPLACED:
					throw new InvalidInstanceChangeException("Can't mark replaced instance as something else. Use revert back instead.");
					
				default:
					throw new InvalidInstanceChangeException("From instance type not recognised.");	
				}
				
				Instance p = fi;
				while((p = p.getPreviousInstance()) != null) {
					if(p.getType() == toType) {
						toAdd = (T) p;
						break;
					}
				}
				toAdd.setInstanceHolder(to);
				tis.add(toAdd);
			}
		}
		
		public void execute() throws InvalidInstanceChangeException {
			if(!executed) {
				to.addInstances(tis);
				from.emptyInstances();
				executed = true;
			}
			else throw new InvalidInstanceChangeException("Cannot execute twice.");
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				from.addInstances(fis);
				to.removeInstances(tis);
				undone = true;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				to.addInstances(tis);
				from.removeInstances(fis);
				undone = false;
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Mark all " + getTypeString(fromType) + " as " + getTypeString(toType) + " (" + to + ")";
		}
		
		public String getActionName() {
			return "Mark all as " + getTypeString(toType);
		}
		
		public int getToType() {
			return toType;
		}
	}

	public static class ReplaceAllFInstancesEdit<F extends VariantInstance> extends AbstractUndoableEdit {
		private static final long serialVersionUID = 1L;
		TreeSet<F> fis;
		TreeSet<ReplacedInstance> ris;
		WordHolder wh;
		int fromType;
		InstanceHolder<F> from;
		InstanceHolder<ReplacedInstance> to;
		
		boolean replaceInText;
		boolean toIsNew = true;
		
		boolean rsHasChangedBefore = confidenceWeights.hasChanged();
		
		boolean executed = false, undone = false;
		
		@SuppressWarnings("unchecked")
		public ReplaceAllFInstancesEdit(WordHolder wh, F tempFrom, SuggestedReplacement rep, boolean replaceInText) throws InvalidInstanceChangeException { //F is basically F(), and is used to get the class type only
			super();
			this.wh = wh;
			fromType = tempFrom.getType();
			this.replaceInText = replaceInText;
			
			switch(fromType) {
			case VARIANT:
				from = (InstanceHolder<F>) wh.asVariant;
				break;
				
			case REAL_ERROR:
				from = (InstanceHolder<F>) wh.asRealError;
				break;
				
			default:
				throw new InvalidInstanceChangeException("From instance type not recognised.");	
			}
			
			to = new InstanceHolder<ReplacedInstance>(wh.word, wh);
			for(HolderChangeListener<ReplacedInstance> hcl : wh.asReplacedChangeListeners) {
				to.addChangeListener(hcl);
			}
			
			for(InstanceHolder<ReplacedInstance> ih : wh.asReplaced) {
				if(!ih.isEmpty() && ih.getInstances().first().getReplacement().equals(rep)) {
					to = ih;
					toIsNew = false;
					break;	
				}
			}
			
			fis = new TreeSet<F>();
			ris = new TreeSet<ReplacedInstance>();
			
			for(F fi : from.getInstances()) {
				fis.add(fi);
				ris.add(new ReplacedInstance(fi, to, rep));
			}
		}
		
		public void execute() throws InvalidInstanceChangeException, BadLocationException {
			if(!executed) {
				for(ReplacedInstance ri : ris) {
					ri.setupInText(replaceInText);
				}
				from.emptyInstances();
				to.addInstances(ris);

				if(toIsNew)
					wh.asReplaced.add(to);
				executed = true;
			}
			else throw new InvalidInstanceChangeException("Cannot execute twice.");
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				try {
					for(ReplacedInstance ri : ris) {
						ri.revertInText();
					}
				}
				catch(BadLocationException e) {
					throw new CannotUndoException();
				}
				to.removeInstances(ris);
				from.addInstances(fis);
				if(toIsNew)
					wh.asReplaced.remove(to);
				
				confidenceWeights.setHasChanged(rsHasChangedBefore);
				undone = true;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				try {
					for(ReplacedInstance ri : ris) {
						ri.setupInText(replaceInText);
					}
				}
				catch(BadLocationException e) {
					throw new CannotRedoException();
				}
				from.removeInstances(fis);
				to.addInstances(ris);
				if(toIsNew)
					wh.asReplaced.add(to);
				undone = false;
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Replace all (" + to + ")";
		}
		
		public String getActionName() {
			return "Replace all";
		}
	}
	
	public static class RevertAllReplacedEdit extends AbstractUndoableEdit {
		private static final long serialVersionUID = 1L;
		TreeSet<ReplacedInstance> ris;
		TreeSet<VariantInstance> vis;
		TreeSet<RealErrorInstance> reis;
		WordHolder wh;
		InstanceHolder<ReplacedInstance> from;
	
		SuggestedReplacement rep;
		
		boolean rsHasChangedBefore = confidenceWeights.hasChanged();
		
		boolean isAlone;
		
		boolean executed = false, undone = false;
		
		public RevertAllReplacedEdit(WordHolder wh, SuggestedReplacement rep) throws InvalidInstanceChangeException { //F is basically F(), and is used to get the class type only
			super();
			this.wh = wh;
			this.rep = rep;
			
			for(InstanceHolder<ReplacedInstance> ih : wh.asReplaced) {
				if(!ih.isEmpty() && ih.getInstances().first().getReplacement().equals(rep)) {
					from = ih;
					if(ih.getInstances().size() == 1)
						isAlone = true;
					break;	
				}
			}
			
			if(from == null)
				throw new InvalidInstanceChangeException("Replacement not found.");
			
			ris = new TreeSet<ReplacedInstance>();
			vis = new TreeSet<VariantInstance>();
			reis = new TreeSet<RealErrorInstance>();
			
			for(ReplacedInstance ri : from.getInstances()) {
				ris.add(ri);
				VariantInstance vi = ri.getPreviousInstance();
				if(vi.getType() == VARIANT)
					vis.add(vi);
				else if(vi.getType() == REAL_ERROR)
					reis.add((RealErrorInstance) vi);
				else throw new InvalidInstanceChangeException("Replaced previous is not variant/RWE.");
			}
		}
		
		public void execute() throws InvalidInstanceChangeException, BadLocationException {
			if(!executed) {
				for(ReplacedInstance ri : ris) {
					ri.revertInText();
				}
				from.emptyInstances();
				wh.asVariant.addInstances(vis);
				wh.asRealError.addInstances(reis);
				if(isAlone)
					wh.asReplaced.remove(from);
				executed = true;
			}
			else throw new InvalidInstanceChangeException("Cannot execute twice.");
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				try {
					for(ReplacedInstance ri : ris) {
						ri.setupInText(true);
					}
				}
				catch(BadLocationException e) {
					throw new CannotUndoException();
				}
				wh.asVariant.removeInstances(vis);
				wh.asRealError.removeInstances(reis);
				from.addInstances(ris);
				if(isAlone)
					wh.asReplaced.add(from);
				
				confidenceWeights.setHasChanged(rsHasChangedBefore);
				
				undone = true;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				try {
					for(ReplacedInstance ri : ris) {
						ri.revertInText();
					}
				}
				catch(BadLocationException e) {
					throw new CannotRedoException();
				}
				from.removeInstances(ris);
				wh.asVariant.addInstances(vis);
				wh.asRealError.addInstances(reis);
				if(isAlone)
					wh.asReplaced.remove(from);
				undone = false;
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Revert all (" + from + ")";
		}
		
		public String getActionName() {
			return "Revert all";
		}
	}

	public int compareTo(WordHolder wh) {
		return word.compareToIgnoreCase(wh.word);
	}
	
	public boolean equals(WordHolder wh) {
		return (compareTo(wh) == 0);
	}

	public VariantInstance addVariant(Position start, Position end, String actual) {
		VariantInstance newInstance = new VariantInstance(start, end, this, asVariant, actual);
		if(!asVariant.addInstance(newInstance)) {
			return (VariantInstance) asVariant.getInstanceAtPos(start.getOffset());
		}
		return newInstance;
	}
	
	public RealErrorInstance addRealError(Position start, Position end, String actual, int reason) {
		RealErrorInstance newInstance = new RealErrorInstance(start, end, this, asRealError, actual, reason);
		if(!asRealError.addInstance(newInstance)) {
			return (RealErrorInstance) asRealError.getInstanceAtPos(start.getOffset());
		}
		return newInstance;
	}
	
	public CorrectInstance addCorrect(Position start, Position end, String actual) {
		CorrectInstance newInstance = new CorrectInstance(start, end, this, asCorrect, actual);
		if(!asCorrect.addInstance(newInstance)) {
			return (CorrectInstance) asCorrect.getInstanceAtPos(start.getOffset());
		}
		return newInstance;
	}
	
	public void addVariant(VariantInstance vi) {
		asVariant.addInstance(vi);
	}
	
	public void addRealError(RealErrorInstance rei) {
		asRealError.addInstance(rei);
	}
	
	public void addCorrect(CorrectInstance ci) {
		asCorrect.addInstance(ci);
	}
	
	public boolean removeVariant(VariantInstance vi) {
		return asVariant.removeInstance(vi);
	}
	
	public boolean removeRealError(RealErrorInstance rei) {
		return asRealError.removeInstance(rei);
	}
	
	public boolean removeCorrect(CorrectInstance ci) {
		return asCorrect.removeInstance(ci);
	}
	
	public Instance getInstanceAt(int caretPos) {
		Instance toReturn;
		if((toReturn = asVariant.getInstanceAtPos(caretPos)) != null)
			return toReturn;
		
		if((toReturn = asRealError.getInstanceAtPos(caretPos)) != null)
			return toReturn;
		
		if((toReturn = asCorrect.getInstanceAtPos(caretPos)) != null)
			return toReturn;
		
		for(InstanceHolder<ReplacedInstance> ih : asReplaced) {
			if((toReturn = ih.getInstanceAtPos(caretPos)) != null)
				return toReturn;
		}
		return null;
	}

	/**
	 * @return the inDictionary
	 */
	public boolean isInDictionary() {
		return inDictionary;
	}

	/**
	 * @return the dictionaryRef
	 */
	public Word getDictionaryRef() {
		return dictionaryRef;
	}

	/**
	 * @return the isLowFreq
	 */
	public boolean isLowFreq() {
		return isLowFreq;
	}

	/**
	 * @param dictionaryRef the dictionaryRef to set
	 */
	public void setDictionaryRef(Word dictionaryRef) {
		this.dictionaryRef = dictionaryRef;
		inDictionary = (dictionaryRef != null);
		if(inDictionary)
			isLowFreq = dictionaryRef.isLowFreq();
		else
			isLowFreq = false;
	}

	/**
	 * @return the word
	 */
	public String getWord() {
		return word;
	}
	
	public boolean areReplacementsSet() {
		return areReplacementsSet;
	}

	/**
	 * @return the replacements
	 */
	public Vector<SuggestedReplacement> getReplacements() {
		return replacements;
	}

	/**
	 * @param replacements the replacements to set
	 */
	public void setReplacements(Vector<SuggestedReplacement> replacements) {
		this.replacements = replacements;
		areReplacementsSet = true;
	}

	/**
	 * @return the soundexCode
	 */
	public String getSoundexCode() {
		return soundexCode;
	}
	
	public SuggestedReplacement getTopReplacement() {
		if(replacements.size()>0)
			return replacements.firstElement();
		else
			return null;
	}
	
	public TreeSet<Instance> getInstances() {
		TreeSet<Instance> toReturn = new TreeSet<Instance>();
		toReturn.addAll(asVariant.getInstances());
		toReturn.addAll(asRealError.getInstances());
		toReturn.addAll(asCorrect.getInstances());
		for(InstanceHolder<ReplacedInstance> ih : asReplaced) {
			toReturn.addAll(ih.getInstances());
		}
		return toReturn;
	}
	
	public void addVariantChangeListener(HolderChangeListener<VariantInstance> hcl) {
		asVariant.addChangeListener(hcl);
	}
		
	public void addCorrectChangeListener(HolderChangeListener<CorrectInstance> hcl) {
		asCorrect.addChangeListener(hcl);
	}
	
	public void addRealErrorChangeListener(HolderChangeListener<RealErrorInstance> hcl) {
		asRealError.addChangeListener(hcl);
	}
		
	public void addReplacedChangeListener(HolderChangeListener<ReplacedInstance> hcl) {
		asReplacedChangeListeners.add(hcl);
		for(InstanceHolder<ReplacedInstance> ih : asReplaced) {
			ih.addChangeListener(hcl);
		}
	}
	
	public TreeSet<SuggestedReplacement> getReplacementsUsed() {
		TreeSet<SuggestedReplacement> toReturn = new TreeSet<SuggestedReplacement>(new SuggestedReplacement.AlphaComparator());
		for(InstanceHolder<ReplacedInstance> ih : asReplaced) {
			ReplacedInstance ri;
			if((ri = ih.firstInstance()) != null)
			toReturn.add(ri.getReplacement());
		}
		return toReturn;
	}
	
	public boolean isAtleastOneInstanceAVariant() {
		return !asVariant.isEmpty();
	}
	
	public boolean isAtleastOneInstanceAReplaced() {
		return !asReplaced.isEmpty();
	}
	
	public boolean isAtleastOneInstanceACorrect() {
		return !asCorrect.isEmpty();
	}
	
	public boolean isAtleastOneInstanceARealError() {
		return !asRealError.isEmpty();
	}
}
