/**
 * 
 */
package doc;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.Position;
import javax.swing.text.StyledDocument;

import lookup.WordUtilities;

/**
 * @author barona
 *
 */
public abstract class Instance implements Comparable<Instance> {
	
	protected Position start, end;
	protected String original, joinString;
	protected boolean isJoin;
	protected int capitalization;
	protected StyledDocument doc;
	protected WordHolder holder;
	protected Instance previousInstance; //marker to what was before present, null if original
	protected InstanceHolder instanceHolder;
	
	public Instance() {
	}
	
	protected Instance(Position start, Position end, WordHolder holder, InstanceHolder instanceHolder, String original) {
		if(start == null)
			throw new NullPointerException("Start null");
		else if(end == null)
			throw new NullPointerException("End null");
		else if(holder == null)
			throw new NullPointerException("WordHolder null");
		else if(instanceHolder == null)
			throw new NullPointerException("InstanceHolder null");
		else if(original == null)
			throw new NullPointerException("Original null");
		else {
			this.start = start;
			this.end = end;
			this.holder = holder;
			this.instanceHolder = instanceHolder;
			this.original = original;
			isJoin = false;
			joinString = null;
			capitalization = WordUtilities.getCapitalization(original);
			doc = holder.getDoc();
			previousInstance = null;
		}
	}
	
	protected Instance(Instance oldInstance) { // for when type changed
		this(oldInstance.start, oldInstance.end, oldInstance.holder, oldInstance.instanceHolder, oldInstance.original);
		previousInstance = oldInstance;
		this.isJoin = previousInstance.isJoin;
		this.joinString = previousInstance.joinString;
	}
	
	public String toString() {
		return original + " (" + getStartOffset() + " - " + getEndOffset() + ")";
	}
	
	
	public abstract String insertXMLTags(int offset, String text);
	
	public abstract int getType();
	
	/**
	 * @return the isJoin
	 */
	public boolean isJoin() {
		return isJoin;
	}

	/**
	 * @param isJoin the isJoin to set
	 */
	public void setJoin(boolean isJoin) {
		this.isJoin = isJoin;
	}

	/**
	 * @return the joinString
	 */
	public String getJoinString() {
		return joinString;
	}

	/**
	 * @param joinString the joinString to set
	 */
	public void setJoinString(String joinString) {
		this.joinString = joinString;
		if(joinString != null) {
			this.joinString = this.joinString.replaceAll("\r\n", "[n]");
			this.joinString = this.joinString.replaceAll("\n", "[n]");
			this.joinString = this.joinString.replaceAll("\r", "[n]");
		}
	}
	
	public int compareTo(Instance another) {
		return new Integer(getStartOffset()).compareTo(new Integer(another.getStartOffset()));
	}
	
	public boolean equals(Instance another) {
		return (compareTo(another) == 0);
	}
	
	public int compareAlphabeticallyTo(Instance other) {
		return original.compareToIgnoreCase(other.original);
	}
	
	public int getStartOffset() {
		return start.getOffset();
	}

	public int getEndOffset() {
		return end.getOffset();
	}

	/**
	 * @return the previousInstance
	 */
	Instance getPreviousInstance() {
		return previousInstance;
	}
	
	void clearPreviousInstance() {
		previousInstance = null;
	}
	
	public boolean containsPosition(int caretPosition) {
		if(caretPosition < getStartOffset())
			return false;
		if(caretPosition < getEndOffset())
			return true;
		return false;
	}
	
	public void changeHighlight(AttributeSet atts) {
		//System.out.println(getStartOffset() + "," + getEndOffset() + " - " + atts);
		doc.setCharacterAttributes(getStartOffset(), getEndOffset()-getStartOffset(), atts, false);
	}

	/**
	 * @return the capitalization
	 */
	public int getCapitalization() {
		return capitalization;
	}

	/**
	 * @return the holder
	 */
	public WordHolder getHolder() {
		return holder;
	}
	
	public abstract boolean removeSelfFromHolder();
	
	public abstract void addSelfToHolder();

	/**
	 * @return Returns the original.
	 */
	public String getOriginal() {
		return original;
	}

	/**
	 * @return the instanceHolder
	 */
	public InstanceHolder getInstanceHolder() {
		return instanceHolder;
	}

	/**
	 * @param instanceHolder the instanceHolder to set
	 */
	public void setInstanceHolder(InstanceHolder instanceHolder) {
		this.instanceHolder = instanceHolder;
	}
	
	//for use when removing and adding back later - i.e. in join
	private int startInt1, endInt1;
	
	//for use when removing and adding back later - i.e. in join
	public void saveStartAndEnd() {
		startInt1 = start.getOffset();
		endInt1 = end.getOffset();
	}
	
	//for use when removing and adding back later - i.e. in join	
	public void restoreStartAndEnd() throws BadLocationException {
		start = doc.createPosition(startInt1);
		end = doc.createPosition(endInt1);
	}
}
