package gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.SpinnerNumberModel;
import javax.swing.SpringLayout;
import javax.swing.Timer;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.Document;
import javax.swing.text.StyledEditorKit;
import javax.swing.text.rtf.RTFEditorKit;

import doc.DocumentModel;


public class BatchScreen implements ActionListener, globals.UserMessageListener, globals.ExceptionMessageHandler {
	
	static final Dimension SCREEN_SIZE = Toolkit.getDefaultToolkit().getScreenSize();

	private JButton[] buttons = {new JButton("Add file(s)"), new JButton("Add folder(s)"), new JButton("Remove selected file(s)"), new JButton("Clear list"), new JButton("Process listed file(s)"), new JButton("Exit"), new JButton("Browse")};
	private JList fileList;
	private Vector<File> files;
	private JScrollPane fileListSP;
	private JTextField outputFolderTF;
	private JProgressBar progressBar;
	private static JLabel progressLabel;
	private JFileChooser fc;
	private JFrame frame;
	private JSpinner thresholdSpinner;
	
	private JCheckBox xmlCheckBox, plainCheckBox;
	
	private File outputFolder;
	
	boolean inputFinished = false;
	int fileCounter = 0;
	private Timer inputTimer;
	String outputFolderString;
	
	private static globals.Globals global;	
	private static DocumentModel docModel;
	
	private BatchScreen() {
		files = new Vector<File>(10,3);
	}
	
	private void createGUI() {

		//input panel
		fileList = new JList(files);
		fileList.setVisibleRowCount(10);
		fileListSP = new JScrollPane(fileList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		fileListSP.setAlignmentX(JScrollPane.LEFT_ALIGNMENT);
		
		for(JButton button : buttons){
			button.addActionListener(this);
		}
		
		JPanel inputButtonsPanel = new JPanel(new GridLayout(6,1));		
		for(int i=0;i<4;i++) {
			inputButtonsPanel.add(buttons[i]);
		}
		inputButtonsPanel.add(Box.createVerticalGlue());
		inputButtonsPanel.add(Box.createVerticalGlue());
		
		JPanel inputPanel = new JPanel(new BorderLayout());
		inputPanel.setBorder(BorderFactory.createTitledBorder("Input"));
		inputPanel.add(fileListSP, BorderLayout.CENTER);
		inputPanel.add(inputButtonsPanel, BorderLayout.EAST);
		
		JPanel thresholdPanel = new JPanel(new SpringLayout());
		JLabel thresholdLabel = new JLabel("Threshold (/100%):", JLabel.TRAILING);
		thresholdPanel.add(thresholdLabel);
		thresholdSpinner = new JSpinner(new SpinnerNumberModel(50, 0, 100, 5));
		thresholdLabel.setLabelFor(thresholdSpinner);
		thresholdPanel.add(thresholdSpinner);


		SpringUtilities.makeCompactGrid(thresholdPanel, 1, 2, 6 , 6, 6, 6);
		
		JPanel threshPanel = new JPanel();
		threshPanel.setBorder(BorderFactory.createTitledBorder("Threshold"));
		threshPanel.add(thresholdPanel);
		
		
		
		//output panel
		
		JPanel outputTypePanel = new JPanel(new GridLayout(2,1));
		plainCheckBox = new JCheckBox("Plain Text");
		xmlCheckBox = new JCheckBox("XML Format (original preserved)");
		
		outputTypePanel.add(plainCheckBox);
		outputTypePanel.add(xmlCheckBox);
		
		
		JPanel outputFilePanel = new JPanel(new SpringLayout());
		JLabel folderLabel = new JLabel("Output Folder:", JLabel.TRAILING);
		outputFilePanel.add(folderLabel);
		outputFolderTF = new JTextField(20);
		folderLabel.setLabelFor(outputFolderTF);
		outputFilePanel.add(outputFolderTF);
		outputFilePanel.add(buttons[6]);
		
		SpringUtilities.makeCompactGrid(outputFilePanel, 1, 3, 6 , 6, 6, 6);
		
	
		JPanel outputPanel = new JPanel(new BorderLayout());
		outputPanel.setBorder(BorderFactory.createTitledBorder("Output"));
		outputPanel.add(outputTypePanel, BorderLayout.CENTER);
		outputPanel.add(outputFilePanel, BorderLayout.SOUTH);
		
		
		//bottom panel
		
		JPanel bottomButtons = new JPanel(new GridLayout(1,2));
		bottomButtons.add(buttons[4]);
		bottomButtons.add(buttons[5]);
		
		progressLabel = new JLabel("");
		progressBar = new JProgressBar(JProgressBar.HORIZONTAL);
		JPanel progressPanel = new JPanel(new GridLayout(2,1));
		progressPanel.setBorder(BorderFactory.createTitledBorder("Progress"));
		progressPanel.add(progressLabel);
		progressPanel.add(progressBar);
		
		JPanel topPanel = new JPanel(new BorderLayout());
		topPanel.add(inputPanel, BorderLayout.CENTER);
		topPanel.add(threshPanel, BorderLayout.SOUTH);
		
		JPanel bottomPanel = new JPanel(new BorderLayout());
		bottomPanel.add(progressPanel, BorderLayout.CENTER);
		bottomPanel.add(bottomButtons, BorderLayout.SOUTH);
		
		fc = new JFileChooser();
		
		frame = new JFrame("VARD 2 Batch Process");
		frame.getContentPane().add(topPanel, BorderLayout.NORTH);
		frame.getContentPane().add(outputPanel, BorderLayout.CENTER);
		frame.getContentPane().add(bottomPanel, BorderLayout.SOUTH);
				
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.pack();
		frame.setLocation(SCREEN_SIZE.width/2 - frame.getWidth()/2, SCREEN_SIZE.height/2 - frame.getHeight()/2);
		frame.setResizable(false);
		frame.setVisible(true);
	}
	
	
	/**
	 * @param args
	 */
	public static void run() {
		global = globals.Globals.getInstance();
		
		new Thread(new Runnable() {
			public void run() {
				global.processingMessager.writeMessage("Loading user interface...");
				BatchScreen ui = new BatchScreen();
				ui.createGUI();
				global.processingMessager.finishMessages();		
				
				global.processingMessager = ui;
				global.exceptionHandler = ui;
				
				docModel = new DocumentModel();
			}
		}).start();
	}
	
	public void actionPerformed(ActionEvent a) {
		int buttonNumber = -1;
		for(int i=0;i<buttons.length;i++) {
			if(a.getSource().equals(buttons[i]))
					buttonNumber = i;
		}
		
		/*
		 * 0 = add file(s)
		 * 1 = add folder(s)
		 * 2 = remove selected files
		 * 3 = clear list
		 * 4 = process list
		 * 5 = exit
		 * 6 = browse
		 * 
		 */
		
		switch(buttonNumber) {
		
		case 0:
			fc.setFileSelectionMode(JFileChooser.FILES_ONLY);
			fc.setMultiSelectionEnabled(true);
			int returnVal = fc.showOpenDialog(frame);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				File[] filesSelected = fc.getSelectedFiles();
				for(int i=0;i<filesSelected.length;i++) {
					if(!files.contains(filesSelected[i]))
						files.add(filesSelected[i]);
				}
			}
			fileList.setListData(files);
			break;
			
		case 1:
			fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			fc.setMultiSelectionEnabled(true);
			int returnVal2 = fc.showOpenDialog(frame);
			if(returnVal2 == JFileChooser.APPROVE_OPTION) {
				File[] directoriesSelected = fc.getSelectedFiles();
				for(int i=0;i<directoriesSelected.length;i++) {
					File[] filesFromFolder = directoriesSelected[i].listFiles();
					for(int j=0;j<filesFromFolder.length;j++) {
						if(!filesFromFolder[j].isDirectory() && !files.contains(filesFromFolder[j]))
							files.add(filesFromFolder[j]);
					}
				}
			}
			
			fileList.setListData(files);
			break;
		
		case 2:
			Object[] selectedFiles = fileList.getSelectedValues();
			for(int i=0;i<selectedFiles.length;i++) {
				files.remove(selectedFiles[i]);
			}
					
			fileList.setListData(files);
			break;
		
		case 3:
			files.removeAllElements();
			fileList.setListData(files);
			break;
			
		case 4: //process list
			outputFolderString = outputFolderTF.getText();
			
			inputFinished = false;

			if(files.size() == 0) {
				displayError("Please provide atleast one file to process.");
				return;
			}
			
			if(outputFolderString.equals("")) {
				displayError("Please provide an output folder.");
				return;
			}
			
		
			setButtonsEnabled(false);
			
			inputTimer = new Timer(50, new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
					progressBar.setMaximum(docModel.getHighestPosToRead());
					progressBar.setValue(docModel.getCurrentPosBeingRead());
					if(inputFinished) {
						inputTimer.stop();
						progressLabel.setText("Completed.");
						progressBar.setValue(0);
						setButtonsEnabled(true);
					}
				}
			});
			
			Thread thread = new Thread(new Runnable() {
				public void run() {
					Integer thresholdInt = (Integer) thresholdSpinner.getValue();
					Double threshold = thresholdInt.doubleValue();
					
					PrintWriter statsWriter;
					File statsFile = new File(outputFolderString + File.separator + "varded(" + thresholdInt.intValue() + "%) stats.txt");
					try {
						statsFile.createNewFile();
						statsWriter = new PrintWriter(new FileWriter(statsFile));
						statsWriter.println("File\tTotal Words\tVariants Forms\tReplaced\tModern Forms\tUncommon words");
					}
					catch(IOException ex) {
						global.exceptionHandler.showException("An error occurred creating stats file.", ex);
						inputTimer.stop();
						progressLabel.setText("Error occurred creating stats file");
						progressBar.setValue(0);
						setButtonsEnabled(true);
						return;
					}

					int totalFiles = files.size();
					
					fileCounter = 0;					
					for(File file : files) {
						try {
							fileCounter++;
							String fileProgress = "Processing " + file.getName() + " (" + fileCounter + "/" + totalFiles + ")";
							progressLabel.setText(fileProgress);
						
							docModel = new DocumentModel();
							Document d = new DefaultStyledDocument();
							JTextPane textPane = new JTextPane();
							if(file.getName().endsWith("rtf")) {
								RTFEditorKit rek = new RTFEditorKit();
								rek.read(new java.io.FileInputStream(file), d, 0);
								textPane.setDocument(d);
							}

							else  {
								StyledEditorKit sek = new StyledEditorKit();
								sek.read(new java.io.FileInputStream(file), d, 0);
								textPane.setDocument(d);
							}
							
							docModel.setDocument(textPane.getStyledDocument());
							progressLabel.setText(fileProgress + ": Evaluating words...");
							inputTimer.start();
							docModel.readWords();
							
							progressLabel.setText(fileProgress + ": Finding replacements for variants...");
							docModel.getReplacements();
							while(!docModel.isReadFinished()) {
								//System.out.print(".");
							}
							
							progressLabel.setText(fileProgress + ": Replacing variants...");
							docModel.getProcessAllVariantsEdit(threshold, false).execute();

							if(plainCheckBox.isSelected()) {
								progressLabel.setText(fileProgress + ": Writing text file.");
								progressBar.setIndeterminate(true);
								File newFile = new File(outputFolderString + File.separator + "varded(" + thresholdInt.intValue() + "%)- " + file.getName());
								newFile.createNewFile();
								docModel.saveToFile(newFile);
								progressBar.setIndeterminate(false);
							}
							if(xmlCheckBox.isSelected()) {
								progressLabel.setText(fileProgress + ": Writing XML file.");
								File newFile = new File(outputFolderString + File.separator + "varded(" + thresholdInt.intValue() + "%)- " + file.getName() + ".xml");
								newFile.createNewFile();
								docModel.exportToXML(newFile);
							}
							statsWriter.println(file.getName() + "\t" + docModel.getStats());
						} 
						catch (Exception e) {
							global.exceptionHandler.showException("An error occurred reading file.", e);
							inputTimer.stop();
							progressLabel.setText("Error occurred reading file: " + file);
							progressBar.setValue(0);
							setButtonsEnabled(true);
							return;
						}
					}
			
					inputFinished = true;
					statsWriter.close();
				}
			});
			thread.start();
			break;
			
		case 5:
			System.exit(0);
			break;
			
		case 6:
			fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			fc.setMultiSelectionEnabled(false);
			int saveReturnVal = fc.showSaveDialog(frame);
			if(saveReturnVal == JFileChooser.APPROVE_OPTION) {
				outputFolder = fc.getSelectedFile();
				outputFolderTF.setText(outputFolder.getPath());
			}
			break;
			
		}
	}
	
	private void setButtonsEnabled(boolean e) {
		buttons[0].setEnabled(e);
		buttons[1].setEnabled(e);
		buttons[2].setEnabled(e);
		buttons[3].setEnabled(e);
		buttons[4].setEnabled(e);
		buttons[6].setEnabled(e);
		thresholdSpinner.setEnabled(e);
		outputFolderTF.setEditable(e);
		xmlCheckBox.setEnabled(e);
		plainCheckBox.setEnabled(e);
	}

	public void finishMessages() {
		progressLabel.setText("");
		setButtonsEnabled(true);
		
	}

	public void writeMessage(String message) {
		setButtonsEnabled(false);
		progressLabel.setText(message);
	}
	
	public static void displayError(String message) {
		JOptionPane.showMessageDialog(MainScreen.frame, message, "Error", JOptionPane.ERROR_MESSAGE);
	}
	
	public void showException(String message, Exception ex) {
		JOptionPane.showMessageDialog(MainScreen.frame, message + "\n\n" + ex.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
		ex.printStackTrace();
	}
	
}