/**
 * 
 */
package doc;

import java.util.TreeSet;
import java.util.Vector;

import lookup.ConfidenceWeights;
import lookup.SuggestedReplacement;
import lookup.WordUtilities;

/**
 * @author barona
 *
 */
public class InstanceHolder<E extends Instance> implements Comparable<InstanceHolder<E>> {
	
	private Vector<HolderChangeListener<E>> changeListeners;
	private TreeSet<E> instances;
	private String word;
	
	private WordHolder wordHolder;
	
	private ConfidenceWeights.ScoreAdjustment confidenceWeightsAdjustment;
	
	InstanceHolder(String word, WordHolder wordHolder) {
		this.word = word;
		this.changeListeners = new Vector<HolderChangeListener<E>>();
		instances = new TreeSet<E>();
		this.wordHolder = wordHolder;
		confidenceWeightsAdjustment = null;
	}
	
	public boolean contains(Instance instance) {
		return instances.contains(instance);
	}
	
	public int getCapitalization() {
		if(isEmpty())
			return WordUtilities.ALL_CAPITALS;
		if(areAllWordsExactlySame()) {
			return instances.first().getCapitalization();
		}
		else if(areAllCapitalizationType(WordUtilities.FIRST_CAPITAL))
			return WordUtilities.FIRST_CAPITAL;
		else if(areAllCapitalizationType(WordUtilities.ALL_CAPITALS))
			return WordUtilities.ALL_CAPITALS;
		else
			return WordUtilities.NO_CAPITALS;
	}
	
	public String toString() {
		if(isEmpty())
			return "0 instances error";
		if(areAllWordsExactlySame()) {
			word = instances.first().getOriginal();
		}
		else if(areAllCapitalizationType(WordUtilities.FIRST_CAPITAL))
			word = WordUtilities.capitalize(word, WordUtilities.FIRST_CAPITAL);
		else if(areAllCapitalizationType(WordUtilities.ALL_CAPITALS))
			word = WordUtilities.capitalize(word, WordUtilities.ALL_CAPITALS);
		else
			word = WordUtilities.capitalize(word, WordUtilities.NO_CAPITALS);
		
		if(instances.first().getType() == WordHolder.REPLACED) {
			SuggestedReplacement rep = ((ReplacedInstance) instances.first()).getReplacement();
			return word + " -> " + rep.getReplacementString() + " (" + instances.size() + ")";
		}
		return word + " (" + instances.size() + ")";
	}
	
	public String toRevertString() { //returns same as toString if not replace.
		if(isEmpty())
			return "0 instances error";
		if(areAllWordsExactlySame()) {
			word = instances.first().getOriginal();
		}
		else if(areAllCapitalizationType(WordUtilities.FIRST_CAPITAL))
			word = WordUtilities.capitalize(word, WordUtilities.FIRST_CAPITAL);
		else if(areAllCapitalizationType(WordUtilities.ALL_CAPITALS))
			word = WordUtilities.capitalize(word, WordUtilities.ALL_CAPITALS);
		else
			word = WordUtilities.capitalize(word, WordUtilities.NO_CAPITALS);
		
		if(instances.first().getType() == WordHolder.REPLACED) {
			SuggestedReplacement rep = ((ReplacedInstance) instances.first()).getReplacement();
			return word + " <- " + rep.getReplacementString() + " (" + instances.size() + ")";
		}
		return word + " (" + instances.size() + ")";
		
	}
	
	public String tabbedString() {
		if(isEmpty())
			return "";
		if(areAllCapitalizationType(WordUtilities.FIRST_CAPITAL))
			word = WordUtilities.capitalize(word, WordUtilities.FIRST_CAPITAL);
		else if(areAllCapitalizationType(WordUtilities.ALL_CAPITALS))
			word = WordUtilities.capitalize(word, WordUtilities.ALL_CAPITALS);
		else
			word = WordUtilities.capitalize(word, WordUtilities.NO_CAPITALS);
	
		if(instances.first().getType() == WordHolder.REPLACED) {
			SuggestedReplacement rep = ((ReplacedInstance) instances.first()).getReplacement();
			return ("\r" + rep.getOldWord() + "\t" + rep.getReplacementString() + "\t" + instances.size() + "\t" + rep.isKnownVariant() + "\t" + rep.isLetterReplacement() + "\t" + rep.isSoundex() + "\t" + rep.getEditDistance());
		}
		return word + "\t" + instances.size();
	}
	
	private boolean areAllWordsExactlySame() {
		String w = instances.first().getOriginal();
		for(Instance i : instances) {
			if(!i.getOriginal().equals(w)) {
				return false;
			}
		}
		return true;
	}
	
	private boolean areAllCapitalizationType(int capType) {
		for(Instance i : instances) {
			if(i.capitalization != capType)
				return false;
		}
		return true;
	}
	
	private void fireEmptiedEvent(E lastRemoved) {
		for(HolderChangeListener<E> hcl : changeListeners) {
			hcl.holderEmptied(this, lastRemoved);
		}
	}
	
	private void fireFilledEvent(E lastAdded) {
		for(HolderChangeListener<E> hcl : changeListeners) {
			hcl.holderFilled(this, lastAdded);
		}
	}
	
	private void fireListChangedEvent() {
		for(HolderChangeListener<E> hcl : changeListeners) {
			hcl.listChanged(this);
		}
	}
	
	private void fireInstanceAboutToChangeEvent(E instance) {
		for(HolderChangeListener<E> hcl : changeListeners) {
			hcl.instanceAboutToChange(instance);
		}
	}
	
	protected void fireInstanceChangedEvent(E instance) {
		for(HolderChangeListener<E> hcl : changeListeners) {
			hcl.instanceChanged(instance);
		}
	}
	
	boolean removeInstance(E toRemove) { //returns true if removed
		fireInstanceAboutToChangeEvent(toRemove);
		if(instances.remove(toRemove)) {
			if(instances.isEmpty())
				fireEmptiedEvent(toRemove);
			fireListChangedEvent();			
			return true;
		}
		else
			return false;			
	}
	
	boolean addInstance(E toAdd) {
		boolean fireFilled = false;
		if(instances.isEmpty())
			fireFilled = true;
		
		if(instances.add(toAdd)) {		
			if(fireFilled)
				fireFilledEvent(toAdd);
			fireInstanceChangedEvent(toAdd);
			fireListChangedEvent();
			return true;
		}
		return false;
	}
	
	TreeSet<E> getInstances() {
		return instances;
	}
	
	void emptyInstances() {
		if(!instances.isEmpty()) {
			for(E instance : instances) {
				fireInstanceAboutToChangeEvent(instance);
			}
			E lastRemoved = instances.first();
			instances.clear();
			fireEmptiedEvent(lastRemoved);
			fireListChangedEvent();
		}
	}
	
	boolean addInstances(TreeSet<E> newInstances) {
		if(!newInstances.isEmpty()) {
			boolean fireFilled = false;
			if(instances.isEmpty())
				fireFilled = true;
		
			if(!newInstances.isEmpty()) {
				if(instances.addAll(newInstances)) {
					if(fireFilled)
						fireFilledEvent(instances.last());
					for(E instance : newInstances) {
						fireInstanceChangedEvent(instance);
					}
					fireListChangedEvent();
					return true;
				}
			}
		}
		return false;
		
	}
	
	void removeInstances(TreeSet<E> toRemove) {
		if(!toRemove.isEmpty()) {
			for(E instance : toRemove) {
				fireInstanceAboutToChangeEvent(instance);
			}
			E lastRemoved = toRemove.first();
			instances.removeAll(toRemove);
			if(instances.isEmpty())
				fireEmptiedEvent(lastRemoved);
			fireListChangedEvent();
		}
	}

	public int compareTo(InstanceHolder<E> ih) {
		if(instances.isEmpty()) {
			if(ih.instances.isEmpty())
				return 0;
			else
				return -1;
		}
		else if(ih.instances.isEmpty())
			return 1;
		else {
			return instances.first().compareAlphabeticallyTo(ih.instances.first());
		}
	}
	
	public static class AlphaComparator<E extends Instance> implements java.util.Comparator<InstanceHolder<E>> {
		public AlphaComparator() {
		}

		public int compare(InstanceHolder<E> ih1, InstanceHolder<E> ih2) {
			int c = ih1.word.compareToIgnoreCase(ih2.word);
			if(c==0)
				return ih1.compareTo(ih2);
			else
				return c;
	
		}
	}
	
	
	
	public boolean isEmpty() {
		return instances.isEmpty();
	}
	
	public E getInstanceAtPos(int caretPos) {
		for(E i : instances) {
			if(i.containsPosition(caretPos))
				return i;
			else if(i.getStartOffset() > caretPos)
				return null;
		}
		return null;
	}

	/**
	 * @param selected the selected to set
	 */
	public void resetAllInstancesHighlights() {
		fireAllInstancesAboutToChange();
		fireAllInstancesChanged();
	}
	
	private void fireAllInstancesAboutToChange() {
		for(E instance : instances) {
			fireInstanceAboutToChangeEvent(instance);
		}
	}
	
	private void fireAllInstancesChanged() {
		for(E instance : instances) {
			fireInstanceChangedEvent(instance);
		}
	}
	
	public E firstInstance() {
		if(instances.isEmpty())
			return null;
		return instances.first();
	}
	
	public int getType() { //returns -1 if no instances yet.
		if(instances.isEmpty())
			return -1;
		else
			return instances.first().getType();
	}
	
	public void addChangeListener(HolderChangeListener<E> hcl) {
		changeListeners.add(hcl);
	}

	/**
	 * @return Returns the wordHolder.
	 */
	public WordHolder getWordHolder() {
		return wordHolder;
	}

	/**
	 * @return the ratingScoresAdjustment
	 */
	public ConfidenceWeights.ScoreAdjustment getConfidenceWeightsAdjustment() {
		return confidenceWeightsAdjustment;
	}

	/**
	 * @param ratingScoresAdjustment the ratingScoresAdjustment to set
	 */
	public void setConfidenceWeightsAdjustment(
			ConfidenceWeights.ScoreAdjustment confidenceWeightsAdjustment) {
		this.confidenceWeightsAdjustment = confidenceWeightsAdjustment;
	}
}
