package doc;


import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.text.StringCharacterIterator;
import java.util.ArrayList;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.Position;
import javax.swing.text.StyledDocument;
import javax.swing.undo.AbstractUndoableEdit;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;

import lookup.ConfidenceWeights;
import lookup.LookUpDictionary;
import lookup.SuggestedReplacement;
import lookup.Word;
import lookup.WordUtilities;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
public class DocumentModel {

	/**
	 * 
	 */
	private static final long serialVersionUID = 6180371772133602545L;
	
	private TreeSet<WordHolder> words;
	
	private StyledDocument doc;
	private Position startPos, endPos;
	private int docLength, highestPosToRead;

	private int currentPosInDoc; //for status of readInWords to monitor progress

	private boolean readFinished;

	private static ConfidenceWeights confidenceWeights;
	private static LookUpDictionary lud;
	
	public int progressBarMax = 0;
	
	private String[] shortCounts = new String[5];
	
	
	private static globals.Globals global;
	
	
	private Vector<HolderChangeListener<VariantInstance>> variantChangeListeners = new Vector<HolderChangeListener<VariantInstance>>();
	private Vector<HolderChangeListener<RealErrorInstance>> realErrorChangeListeners = new Vector<HolderChangeListener<RealErrorInstance>>();
	private Vector<HolderChangeListener<CorrectInstance>> correctChangeListeners = new Vector<HolderChangeListener<CorrectInstance>>();
	private Vector<HolderChangeListener<ReplacedInstance>> replacedChangeListeners = new Vector<HolderChangeListener<ReplacedInstance>>();
	
	public void addVariantChangeListener(HolderChangeListener<VariantInstance> hcl) {
		variantChangeListeners.add(hcl);
		for(WordHolder wh : words) {
			wh.addVariantChangeListener(hcl);
		}
	}
	
	public void addRealErrorChangeListener(HolderChangeListener<RealErrorInstance> hcl) {
		realErrorChangeListeners.add(hcl);
		for(WordHolder wh : words) {
			wh.addRealErrorChangeListener(hcl);
		}
	}
	
	public void addCorrectChangeListener(HolderChangeListener<CorrectInstance> hcl) {
		correctChangeListeners.add(hcl);
		for(WordHolder wh : words) {
			wh.addCorrectChangeListener(hcl);
		}
	}
	
	public void addReplacedChangeListener(HolderChangeListener<ReplacedInstance> rcl) {
		for(WordHolder wh : words) {
			wh.addReplacedChangeListener(rcl);
		}
		replacedChangeListeners.add(rcl);
	}
	
	
	
	public StyledDocument getDoc() {
		return this.doc;
	}
	
	public void setDoc(StyledDocument doc) {
		this.doc = doc;
	}
	
	public Position getStartPos() {
		return this.startPos;
	}
	
	public void setStartPos(Position startPos) {
		this.startPos = startPos;
	}
	
	public Position getEndPos() {
		return this.endPos;
	}
	
	public void setEndPos(Position endPos) {
		this.endPos = endPos;
	}
	
	public int getDocLength() {
		return this.docLength;
	}
	
	public void setDocLength(int docLength) {
		this.docLength = docLength;
	}
	
	public int getHighestPosToBeRead() {
		return this.highestPosToRead;
	}
	
	public void setHighestPosToBeRead(int highestPosToRead) {
		this.highestPosToRead = highestPosToRead;
	}

	public DocumentModel() {
		global = globals.Globals.getInstance();
		lud = LookUpDictionary.getInstance();
		confidenceWeights = ConfidenceWeights.getInstance();
		
		replacedChangeListeners.add(confidenceWeights);
		words = new TreeSet<WordHolder>();
		doc = new DefaultStyledDocument();
		setUpForDocument();
	}
	
	//added 16/7/07
	private WordHolder getReferenceTo(String word) {
		WordHolder newWH = new WordHolder(word, doc);
		if(!words.add(newWH)) {
			for(WordHolder wh : words) {
				if(wh.equals(newWH))
					return wh;
			}
		}
		newWH.setDictionaryRef(lud.checkWords(word));
		addListeners(newWH);
		return newWH;
	}
	
	//added 17/7/07
	public void addListeners(WordHolder wh) {
		for(HolderChangeListener<VariantInstance> hcl : variantChangeListeners) {
			wh.addVariantChangeListener(hcl);
		}
		for(HolderChangeListener<RealErrorInstance> hcl : realErrorChangeListeners) {
			wh.addRealErrorChangeListener(hcl);
		}
		for(HolderChangeListener<CorrectInstance> hcl : correctChangeListeners) {
			wh.addCorrectChangeListener(hcl);
		}
		for(HolderChangeListener<ReplacedInstance> hcl : replacedChangeListeners) {
			wh.addReplacedChangeListener(hcl);
		}
	}
	
//	added 30/8/06 - for replacement analysis
	public String getShortReplacementAnalysis(int line) {
		return shortCounts[line];
	}
	
	
	//added 30/8/06 - for finding how many replacements have been found by each method
	public void countReplacements() {
		int known = 0;
		int lr = 0;
		int soundex = 0;
		int[] editDistances = new int[11];
		int total = 0;
		
		int currentED;
		
/*			Vector<SuggestedReplacement> replacements;
		for(WordInDocument word : replaced) {
			replacements = new Vector<SuggestedReplacement>(2,1);
			for(WordInstanceInDocument instance : word.getInstances()) {
				if(instance.getState()==WordInstanceInDocument.REPLACED && !replacements.contains(instance.getReplacement())) {
					replacements.add(instance.getReplacement());
				}
			}
*/
		
		TreeSet<SuggestedReplacement> replacements = new TreeSet<SuggestedReplacement>(new SuggestedReplacement.AlphaComparator());
		for(WordHolder wh : words) {
			replacements.addAll(wh.getReplacementsUsed());
		}
		
			for(SuggestedReplacement sr : replacements) {
				total++;
				if(sr.isKnownVariant())
					known++;
				if(sr.isLetterReplacement())
					lr++;
				if(sr.isSoundex())
					soundex++;
				currentED = sr.getEditDistance();
				if(currentED > 10)
					currentED=10;
				editDistances[currentED]++;
			}
		
		int EDcount = 0;
		double averageED;
		
		for(int i=0; i<11;i++) {
			EDcount += i*editDistances[i];
		}
		
		if(total!=0)
			averageED = new Double(EDcount)/new Double(total);
		else
			averageED = 0.0;
		
		shortCounts[0] = ("Total: " + total);
		shortCounts[1] = ("Known Var.: " + known);
		shortCounts[2] = ("Letter Repl.: " + lr);
		shortCounts[3] = ("Soundex: " + soundex);
		shortCounts[4] = ("Avg Edit Dist.: " + averageED);
	}
	
	//added 30/8/06 - for finding how many replacements have been found by each method
	public String getReplacementAnalysis() {
		int known = 0;
		int lr = 0;
		int soundex = 0;
		int[] editDistances = new int[11];
		int total = 0;
		
		int currentED;
		
/*			Vector<SuggestedReplacement> replacements;
		for(WordInDocument word : replaced) {
			replacements = new Vector<SuggestedReplacement>(2,1);
			for(WordInstanceInDocument instance : word.getInstances()) {
				if(instance.getState()==WordInstanceInDocument.REPLACED && !replacements.contains(instance.getReplacement())) {
					replacements.add(instance.getReplacement());
				}
			}
*/
		TreeSet<SuggestedReplacement> replacements = new TreeSet<SuggestedReplacement>(new SuggestedReplacement.AlphaComparator());
		for(WordHolder wh : words) {
			replacements.addAll(wh.getReplacementsUsed());
		}
		
			for(SuggestedReplacement sr : replacements) {
				total++;
				if(sr.isKnownVariant())
					known++;
				if(sr.isLetterReplacement())
					lr++;
				if(sr.isSoundex())
					soundex++;
				currentED = sr.getEditDistance();
				if(currentED > 10)
					currentED=10;
				editDistances[currentED]++;
			}
		
		if(total==0)
			return "No analysis possible";
		
		int EDcount = 0;
		double averageED;
		double[] editDistancePercents = new double[11];
		
		for(int i=0; i<11;i++) {
			EDcount += i*editDistances[i];
			editDistancePercents[i] = (new Double(editDistances[i])/new Double(total))*100.00;
		}
		averageED = new Double(EDcount)/new Double(total);
		
		double knownPercent = (new Double(known)/new Double(total))*100.00;
		double lrPercent = (new Double(lr)/new Double(total))*100.00;
		double soundexPercent = (new Double(soundex)/new Double(total))*100.00;
		
		String toReturn = "\tCount\tPercent of total";
		toReturn += ("\rTotal Replacements:\t" + total);
		toReturn += "\rFinding Method Analysis";
		toReturn += ("\rKnown Variant\t" + known + "\t" + knownPercent);
		toReturn += ("\rLetter Replacement\t" + lr + "\t" + lrPercent);
		toReturn += ("\rSoundex\t" + soundex + "\t" + soundexPercent);
		toReturn += "\r";
		toReturn += "\rEdit Distance Analysis";
		for(int i=0;i<11;i++) {
			toReturn += ("\r" + i + " edits\t" + editDistances[i] + "\t" + editDistancePercents[i]);
		}
		toReturn += ("\r\rAverage Edit Distance:\t" + averageED);
		
		return toReturn;
	}
	
	//added 29/8/06 - for copying list to clipboard
/*	private String getCorrectWordsList() {
		String toReturn = "Correct Words\rWord\tFrequency";
		for(WordHolder wh : words) {
			toReturn += (wh.getCorrectStrings());
		}
		return toReturn;
	}
*/

	
	//added 25/6/07 - for copying low freq list to clipboard
/*	private String getLowFreqWordsList() {
		Vector<WordInDocument> lfs= getLowFreqWords();
		Collections.sort(lfs);
		String toReturn = "";
		for(WordInDocument word : lfs) {
			toReturn += (word.tabbedString() + "\r");
		}
		return toReturn;
	}
*/

	//added 29/8/06 - for copying list to clipboard
/*	private String getVariantsList() {
		Vector<WordInDocument> vars = getVariants();
		Collections.sort(vars);
		String toReturn = "";
		for(WordInDocument word : vars) {
			toReturn += (word.tabbedString() + "\r");
		}
		return toReturn;
	}
*/
	
	//added 29/8/06 - for copying list to clipboard
/*	private String getReplacedList() {
		String toReturn = "Variant\tReplacement\tFrequency\tKnown Variant\tLetter Replacement\tSoundex\tEdit Distance";
/*
		Vector<SuggestedReplacement> reps = getReplaced();
		Collections.sort(reps, new SuggestedReplacement.AlphaComparator());
		
		for(SuggestedReplacement rep : reps) {
			toReturn += ("\r" + rep.getOldWord() + "\t" + rep.getReplacementString() + "\t" + rep.getUseCount() + "\t" + rep.isKnownVariant() + "\t" + rep.isLetterReplacement() + "\t" + rep.isSoundex() + "\t" + rep.getEditDistance());
		}
		return toReturn;
	}
*/	
	
	public void setDocument(StyledDocument doc) {
		this.doc = doc;
		readFinished = false;
		setUpForDocument();
	}

	public StyledDocument getDocument() {
		return doc;
	}

	private void setUpForDocument() {
		startPos = doc.getStartPosition();
		endPos = doc.getEndPosition();
		docLength = doc.getLength();
		words = new TreeSet<WordHolder>();
	}
	
	
	public void readXML(String file, boolean updateConfidenceWeights) throws BadLocationException, BadXMLSyntaxException {
		new XMLParser(true, updateConfidenceWeights, file);
	}
	
	public JoinEdit getJoinEdit(int start, int end) throws BadLocationException {
		return new JoinEdit(start, end);
	}
	
	public ProcessAllVariantsEdit getProcessAllVariantsEdit(double threshold, boolean updateConfidenceWeights) {
		return new ProcessAllVariantsEdit(threshold, updateConfidenceWeights);
	}
	
	public class JoinEdit extends AbstractUndoableEdit {

		private static final long serialVersionUID = 1L;
		
		private boolean executed = false, undone = false, valid = false;
		private String whyInvalid = "";
		
		private TreeSet<Instance> wordsSelected;
		private String originalText, newWord;
		private int capitalization;
		private AttributeSet atts;
		
		int start;
		
		private VariantInstance createdVI;
		
		private JoinEdit(int start, int end) throws BadLocationException {
			super();
			
			
			//find all the words in the selection
			wordsSelected = new TreeSet<Instance>();
									
			for(int i=start-1;i<=end;i++) {
				Instance toAdd;
				if((toAdd = getWordAt(i)) != null) {
					if(toAdd.getType() == WordHolder.REPLACED) {
						valid = false;
						whyInvalid = "Cannot join a replaced string.";
						return;
					}
					wordsSelected.add(toAdd);
				}
			}
			
			//check atleast 2 words have been selected
			if(wordsSelected.size() < 2) {
				valid = false;
				whyInvalid = "Atleast 2 words need to be selected to join.";
				return;
			}
			

			//adjust start and end of selection based on words found
			int newStart = wordsSelected.first().getStartOffset();
			int newEnd = wordsSelected.last().getEndOffset();
			
			if(newStart<start)
				start = newStart;
			if(newEnd>end)
				end = newEnd;
			
			this.start = start;
		
			//get properties of text selected
			originalText = doc.getText(start, end-start);
			atts = doc.getCharacterElement(start).getAttributes();
			capitalization = wordsSelected.first().getCapitalization();
			
			//produce new word
			newWord = new String(originalText);
			newWord = newWord.replaceAll("[\\-][^a-zA-Z\']", "");
			newWord = newWord.charAt(0) + newWord.substring(1).replaceAll("[^a-zA-Z\'\\-]", "");
			
			newWord = WordUtilities.capitalize(newWord, capitalization);
			
			//check produced word is valid
			if(!isValidWord(newWord)) {
				valid = false;
				whyInvalid = "Joined word produced is invalid.";
				return;
			}
			valid = true;
		}
		
		public void execute() throws BadLocationException, InvalidInstanceChangeException {
			if(!executed && valid) {				
				for(Instance instance : wordsSelected) {
					instance.saveStartAndEnd();
					instance.removeSelfFromHolder();
				}				
				try {
					doc.remove(start, originalText.length());
				}
				catch(BadLocationException ex) {
//					undo everything done so far - atomic
					for(Instance instance : wordsSelected) {
						instance.addSelfToHolder();
					}
					throw ex;
				}
				try {
					doc.insertString(start, newWord, atts);
				}
				catch(BadLocationException ex) {
//					undo everything done so far - atomic
					doc.insertString(start, originalText, atts);
					for(Instance instance : wordsSelected) {
						instance.restoreStartAndEnd();
						instance.addSelfToHolder();
					}
					throw ex;
				}
				
				try {
					Position joinStartPos = doc.createPosition(start);
					Position joinEndPos = doc.createPosition(start + newWord.length());

					String newWordLC = newWord.toLowerCase();
					WordHolder newWH = getReferenceTo(newWordLC);
					createdVI = newWH.addVariant(joinStartPos, joinEndPos, newWord);
					createdVI.setJoin(true);
					createdVI.setJoinString(originalText);
					createdVI.saveStartAndEnd();
				}
				catch(BadLocationException ex) {
					//undo everything done so far - atomic
					doc.remove(start, newWord.length());
					doc.insertString(start, originalText, atts);
					for(Instance instance : wordsSelected) {
						instance.restoreStartAndEnd();
						instance.addSelfToHolder();
					}
					throw ex;
				}
				
				executed = true;
			}
			else throw new InvalidInstanceChangeException("Join is invalid");
			
		}
		
		
		// TODO check startPos isn't moving for undo and redo
		public void undo() throws CannotUndoException {
			if(canUndo()) {				
				createdVI.saveStartAndEnd();
				createdVI.removeSelfFromHolder();
								
				try {
					doc.remove(start, newWord.length());
				}
				catch(BadLocationException ex) {
					//undo everything done so far - atomic
					createdVI.addSelfToHolder();
					throw new CannotUndoException();
				}
				try {
					doc.insertString(start, originalText, atts);
				}
				catch(BadLocationException ex) {
					//undo everything done so far, atomic
					try {
						doc.insertString(start, newWord, atts);
						createdVI.restoreStartAndEnd();
						createdVI.addSelfToHolder();
						throw new CannotUndoException();
					}
					catch(BadLocationException ex2) {
						throw new CannotUndoException();
					}
				}
				
				try {		
					for(Instance instance : wordsSelected) {
						instance.restoreStartAndEnd();
					}
				}
				catch(BadLocationException ex) {
					//undo everything done so far, atomic
					try {
						doc.remove(start, originalText.length());
						doc.insertString(start, newWord, atts);
						createdVI.restoreStartAndEnd();
						createdVI.addSelfToHolder();
						throw new CannotUndoException();
					}
					catch(BadLocationException ex2) {
						throw new CannotUndoException();
					}
				}
				for(Instance instance : wordsSelected) {
					instance.addSelfToHolder();
				}
				
				undone = true;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(canRedo()) {				
			
				for(Instance instance : wordsSelected) {
					instance.saveStartAndEnd();
					instance.removeSelfFromHolder();
				}
				
				try {
					doc.remove(start, originalText.length());
				}
				catch(BadLocationException ex) {
					//undo everything done so far - atomic
					for(Instance instance : wordsSelected) {
						instance.addSelfToHolder();
					}
					throw new CannotUndoException();
				}
				try {
					doc.insertString(start, newWord, atts);
				}
				catch(BadLocationException ex) {
					//undo everything done so far - atomic
					try {
						doc.insertString(start, originalText, atts);
					}
					catch(BadLocationException ex2) {
						throw new CannotRedoException();						
					}
					for(Instance instance : wordsSelected) {
						try {
							instance.restoreStartAndEnd();
						}
						catch(BadLocationException ex2) {
							throw new CannotRedoException();
						}
						instance.addSelfToHolder();
					}
					throw new CannotRedoException();
				}
				
				try {
					createdVI.restoreStartAndEnd();
				}
				catch(BadLocationException ex) {
					//undo everything done so far, atomic
					try {
						doc.remove(start, newWord.length());
						doc.insertString(start, originalText, atts);
						for(Instance instance : wordsSelected) {
							instance.restoreStartAndEnd();
							instance.addSelfToHolder();
						}
						throw new CannotRedoException();
					}
					catch(BadLocationException ex2) {
						throw new CannotRedoException();
					}
				}
				createdVI.addSelfToHolder();
								
				undone = false;
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed && valid;
		}
		
		public boolean canRedo() {
			return undone && valid;
		}
		
		public String getPresentationName() {
			return "Join words (" + originalText + " -> " + newWord + ")";
		}
		
		public boolean isValid() {
			return valid;
		}
		
		public String getWhyInvalid() {
			return whyInvalid;
		}
	}
	
	public void readWords() throws BadLocationException {
		readFinished = false;
		if(docLength<1) {
			readFinished = true;
			return;
		}
		
		highestPosToRead = docLength;

		int start = startPos.getOffset();
		String text = doc.getText(start, docLength);

		StringCharacterIterator sci = new StringCharacterIterator(text);
		char currentLetter = sci.first();
		char lastLetter;
		int currentPos = sci.getIndex();
		int currentStart = 0;
		int currentEnd = 0;
		currentPosInDoc = 0;
		WordHolder wh;
		String currentWordString, currentWordStringLC;
		boolean inWord = false;
		boolean inTag = false;
		if(isLetter(currentLetter))
			inWord = true;
		else if(currentLetter == '<')
			inTag = true;

		do {
			lastLetter = currentLetter;
			currentLetter = sci.next();
			currentPos = sci.getIndex();

			if(inTag) {
				if(currentLetter == '>')
					inTag = false;
			}			
			else if(inWord) {
				if(!isLetter(currentLetter) && !(currentLetter == '-')) {
					if(lastLetter == '-')
						currentEnd = currentPos-1;
					else
						currentEnd = currentPos;

					if((currentEnd-currentStart)>0) {
						currentWordString = text.substring(currentStart,currentEnd);
						if(isValidWord(currentWordString)) {
							currentWordStringLC = currentWordString.toLowerCase();
							wh = getReferenceTo(currentWordStringLC);
							if(wh.isInDictionary()) {
								if(wh.isLowFreq()) {
									wh.addRealError(doc.createPosition(start+currentStart), doc.createPosition(start+currentEnd), currentWordString, RealErrorInstance.LOW_FREQ);
									//System.out.print("u");
								}
								else {
									wh.addCorrect(doc.createPosition(start+currentStart), doc.createPosition(start+currentEnd), currentWordString);
									//System.out.print("c");
								}
							}
							else {
								wh.addVariant(doc.createPosition(start+currentStart), doc.createPosition(start+currentEnd), currentWordString);
								//findReplacementsForWord(wh);
								//System.out.print("v");
							}
						}
					}
					inWord = false;
				}
			}
			else if(isLetter(currentLetter)) {
				currentStart = currentPos;
				inWord = true;
			}
			else if(currentLetter == '<')
				inTag = true;
			
			currentPosInDoc++; //for progress check

		} while(currentLetter != StringCharacterIterator.DONE);

		readFinished = true;
	}

	public void insertTextAndProcess(String text, AttributeSet textStyle) throws BadLocationException {
		readFinished = false;

		if(text.length()<1) {
			readFinished = true;
			return;
		}

		if(doc.getLength() > 0)
			doc.insertString(endPos.getOffset(), "\n\n", textStyle);


		int start = endPos.getOffset()-1;

		doc.insertString(start, text, textStyle);

		StringCharacterIterator sci = new StringCharacterIterator(text);
		char currentLetter = sci.first();
		char lastLetter;
		int currentPos = sci.getIndex();
		int currentStart = 0;
		int currentEnd = 0;
		currentPosInDoc = 0;
		WordHolder wh;
		String currentWordString, currentWordStringLC;
		boolean inWord = false;
		boolean inTag = false;
		if(isLetter(currentLetter))
			inWord = true;
		else if(currentLetter == '<')
			inTag = true;

		do {
			lastLetter = currentLetter;
			currentLetter = sci.next();
			currentPos = sci.getIndex();

			if(inTag) {
				if(currentLetter == '>')
					inTag = false;
			}			
			else if(inWord) {
				if(!isLetter(currentLetter) && !(currentLetter == '-')) {
					if(lastLetter == '-')
						currentEnd = currentPos-1;
					else
						currentEnd = currentPos;

					if((currentEnd-currentStart)>0) {
						currentWordString = text.substring(currentStart,currentEnd);
						if(isValidWord(currentWordString)) {
							currentWordStringLC = currentWordString.toLowerCase();
							wh = getReferenceTo(currentWordStringLC);
							if(wh.isInDictionary()) {
								if(wh.isLowFreq())
									wh.addRealError(doc.createPosition(start+currentStart), doc.createPosition(start+currentEnd), currentWordString, RealErrorInstance.LOW_FREQ);
								else
									wh.addCorrect(doc.createPosition(start+currentStart), doc.createPosition(start+currentEnd), currentWordString);
							}
							else
								wh.addVariant(doc.createPosition(start+currentStart), doc.createPosition(start+currentEnd), currentWordString);
						}
					}
					inWord = false;
				}
			}
			else if(isLetter(currentLetter)) {
				currentStart = currentPos;
				inWord = true;
			}
			else if(currentLetter == '<')
				inTag = true;
			
			currentPosInDoc++; //for progress check

		} while(currentLetter != StringCharacterIterator.DONE);

		readFinished = true;
	}
		
	public boolean isLetter(char letter) {
		if(letter == '\'' || letter == '' || letter == '')
			return true;
		else if(Character.isDigit(letter) == true)
			return true;
		else
			return Character.isLetter(letter);
	}

	public boolean isValidWord(String word) {
		if(word.contains("--"))
			return false;
		char[] array = word.toCharArray();
		for(int i=0;i<array.length;i++) {
			if(!lookup.WordUtilities.isLetter(array[i]) && array[i] != '-')
				return false;
		}
		return true;
	}

	public int getCurrentPosBeingRead() {
		return currentPosInDoc;
	}

	public boolean isReadFinished() {
		return readFinished;
	}

	public Instance getWordAt(int caretPos) {
		Instance toReturn;
		for(WordHolder wh : words) {
			if((toReturn = wh.getInstanceAt(caretPos)) != null)
				return toReturn;
		}
		return null;
	}
	
	public static class AddWordToDictionaryEdit extends AbstractUndoableEdit {
		private static final long serialVersionUID = 1L;
		boolean ludWordsChangedBefore = lud.isWordsChanged();
		String ludWordsLogBefore = lud.getWordsLog();
		boolean executed = false, undone = false;
		
		WordHolder wh;
		String word;
		
		public AddWordToDictionaryEdit(WordHolder wh, String word) {
			this.wh = wh;
			this.word = word;
		}
		
		public void execute() {
			if(!executed) {
				wh.setDictionaryRef(addWordToDictionary(word));
				executed = true;
			}
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				if(lud.removeWord(word))
					wh.setDictionaryRef(null);
				lud.setWordsChanged(ludWordsChangedBefore, null);
				lud.setWordsLog(ludWordsLogBefore);
				
				undone = true;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				wh.setDictionaryRef(addWordToDictionary(word));
				undone = false;
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Add " + word + " to dictionary.";
		}
	}

	public static class RemoveWordFromDictionaryEdit extends AbstractUndoableEdit {
		private static final long serialVersionUID = 1L;
		boolean ludWordsChangedBefore = lud.isWordsChanged();
		String ludWordsLogBefore = lud.getWordsLog();
		boolean executed = false, undone = false;
		
		WordHolder wh;
		String word;
		
		public RemoveWordFromDictionaryEdit(WordHolder wh, String word) {
			this.wh = wh;
			this.word = word;
		}
		
		public void execute() {
			if(!executed) {
				if(lud.removeWord(word)) {
					lud.setWordsChanged(true, globals.Globals.NEW_LINE + "removed: "+ word);
					wh.setDictionaryRef(null);
				}
				executed = true;
			}
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				wh.setDictionaryRef(addWordToDictionary(word));
				lud.setWordsChanged(ludWordsChangedBefore, null);
				lud.setWordsLog(ludWordsLogBefore);
				undone = true;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				if(lud.removeWord(word)) {
					lud.setWordsChanged(true, globals.Globals.NEW_LINE + "removed: "+ word);
					wh.setDictionaryRef(null);
				}
				undone = false;
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Remove " + word + " from dictionary.";
		}
	}

	private static Word addWordToDictionary(String word) {
		if(lud.checkWords(word.toString()) == null) {
			lud.setWordsChanged(true, globals.Globals.NEW_LINE + "added: " + word);
		}
		return lud.addUserWord(word.toString());
	}
	
	public static class AddVariantToLookUpEdit extends AbstractUndoableEdit {
		private static final long serialVersionUID = 1L;
		boolean ludVariantsChangedBefore = lud.isVariantsChanged();
		String ludVariantsChangeLog = lud.getVariantsLog();
		boolean executed = false, undone = false;
		
		SuggestedReplacement rep;
		
		public AddVariantToLookUpEdit(SuggestedReplacement rep) {
			this.rep = rep;
		}
		
		public void execute() {
			if(!executed) {
				if(lud.addUserVariant(rep.getOldWord(), rep.getReplacement())) {
					lud.setVariantsChanged(true, globals.Globals.NEW_LINE + "added: "+ rep.getOldWord() + " -> " + rep.getReplacementString());
				}
				executed = true;
			}
		}
		
		public void undo() throws CannotUndoException {
			if(executed) {
				lud.removeVariant(rep.getOldWord());
				lud.setVariantsChanged(ludVariantsChangedBefore, null);
				lud.setVariantsLog(ludVariantsChangeLog);
				undone = true;
			}
			else throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(undone) {
				if(lud.addUserVariant(rep.getOldWord(), rep.getReplacement())) {
					lud.setVariantsChanged(true, globals.Globals.NEW_LINE + "added: "+ rep.getOldWord() + " -> " + rep.getReplacementString());
				}
				undone = false;
			}
			else throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Add " + rep.getOldWord() + " -> " + rep.getReplacementString() + " to known variants list.";
		}
	}

	public class ProcessAllVariantsEdit extends ThreadableEdit {

		private static final long serialVersionUID = 1L;
		
		private boolean executed, undone, update, processComplete, hasChangedBefore = confidenceWeights.hasChanged();
		private double threshold;
		
		private int currentPos;
		
		private ArrayList<WordHolder.ReplaceAllFInstancesEdit<VariantInstance>> edits, editsUndone, editsRedone;
		
		private ProcessAllVariantsEdit(double threshold, boolean updateConfidenceWeights) {
			super();
			this.threshold = threshold;
			this.update = updateConfidenceWeights;
			edits = new ArrayList<WordHolder.ReplaceAllFInstancesEdit<VariantInstance>>();
		}
		
		public void execute() throws CannotExecuteException {
			if(!executed) {
				processComplete = false;
				confidenceWeights.setUpdate(update);
				currentPos = 0;
				try {
					for(WordHolder wh : words) {
						SuggestedReplacement top = wh.getTopReplacement();
						if(top!=null && top.getScoreWithoutUpdating()>=threshold) {
							WordHolder.ReplaceAllFInstancesEdit<VariantInstance> edit = new WordHolder.ReplaceAllFInstancesEdit<VariantInstance>(wh, new VariantInstance(), top, true);
							edit.execute();
							edits.add(edit);
						}
						currentPos++;
					}
				}
				catch(InvalidInstanceChangeException ex) {
					//atomic
					for(WordHolder.ReplaceAllFInstancesEdit<VariantInstance> edit : edits) {
						edit.undo();
					}
					throw new CannotExecuteException(ex.getMessage());
				}
				catch(BadLocationException ex) {
					//atomic
					for(WordHolder.ReplaceAllFInstancesEdit<VariantInstance> edit : edits) {
						edit.undo();
					}
					throw new CannotExecuteException(ex.getMessage());
				}
				finally {
					confidenceWeights.setUpdate(true);
					processComplete = true;
				}
				
				executed = true;
				
			}
		}
		
		public void undo() throws CannotUndoException {
			if(canUndo()) {
				processComplete = false;
				editsUndone = new ArrayList<WordHolder.ReplaceAllFInstancesEdit<VariantInstance>>();
				ArrayList<WordHolder.ReplaceAllFInstancesEdit<VariantInstance>> toUse = edits;
				if(editsRedone != null && !editsRedone.isEmpty())
					toUse = editsRedone;
				confidenceWeights.setUpdate(update);
				confidenceWeights.setHasChanged(hasChangedBefore);
				currentPos = 0;
				try {
					for(WordHolder.ReplaceAllFInstancesEdit<VariantInstance> edit : toUse) {
						edit.undo();
						editsUndone.add(edit);
						currentPos++;
					}
				}
				catch(CannotUndoException ex) {
					//atomic
					for(WordHolder.ReplaceAllFInstancesEdit<VariantInstance> edit : editsUndone) {
						edit.redo();
					}
					throw ex;
				}
				
				finally {
					confidenceWeights.setUpdate(true);
					processComplete = true;
				}
				
				undone = true;
			}
			else
				throw new CannotUndoException();
		}
		
		public void redo() throws CannotRedoException {
			if(canRedo()) {
				processComplete = false;
				editsRedone = new ArrayList<WordHolder.ReplaceAllFInstancesEdit<VariantInstance>>();
				confidenceWeights.setUpdate(update);
				currentPos = 0;
				try {
					for(WordHolder.ReplaceAllFInstancesEdit<VariantInstance> edit : editsUndone) {
						edit.redo();
						editsRedone.add(edit);
						currentPos++;
					}
				}
				catch(CannotRedoException ex) {
					//atomic
					for(WordHolder.ReplaceAllFInstancesEdit<VariantInstance> edit : editsRedone) {
						edit.undo();
					}
					throw ex;
				}
				
				finally {
					confidenceWeights.setUpdate(true);
					processComplete = true;
				}
				
				undone = false;
			}
			else
				throw new CannotRedoException();
		}
		
		public boolean canUndo() {
			return executed;
		}
		
		public boolean canRedo() {
			return undone;
		}
		
		public String getPresentationName() {
			return "Process all variants (" + threshold + "%)";
		}

		@Override
		public int getCurrentPos() {
			return currentPos;
		}

		@Override
		public int getRedoSize() {
			if(editsUndone != null)
				return editsUndone.size();
			else
				return 0;
		}

		@Override
		public int getUndoSize() {
			ArrayList<WordHolder.ReplaceAllFInstancesEdit<VariantInstance>> toUse = edits;
			if(editsRedone != null && !editsRedone.isEmpty())
				toUse = editsRedone;
			
			if(toUse != null)
				return toUse.size();
			else
				return 0;
		}

		@Override
		public boolean isProcessComplete() {
			return processComplete;
		}

		@Override
		public int getExecuteSize() {
			return words.size();
		}
		
	}

	public int getWordsSize() {
		return words.size();
	}

	public void saveToFile(final File file) {
		new Thread(new Runnable() {
			public void run() {
				try {
					if(file.getName().endsWith("rtf"))
						new javax.swing.text.rtf.RTFEditorKit().write(new FileOutputStream(file), doc, 0, doc.getLength());	
					else
						new javax.swing.text.DefaultEditorKit().write(new FileOutputStream(file), doc, 0, doc.getLength());
				}
				catch (BadLocationException e) {
					global.exceptionHandler.showException("Error occurred saving document.", e);
				} catch (IOException e) {
					global.exceptionHandler.showException("Error occurred saving document.", e);
				}
			}
		}).start();
	}

	public void exportToXML(File file) throws IOException, BadLocationException {
		readFinished = false;
		//DefaultStyledDocument d = new DefaultStyledDocument();
		//d.insertString(0, getXMLTaggedText(), null);
		//new javax.swing.text.DefaultEditorKit().write(new FileOutputStream(file), d, 0, d.getLength());
		PrintWriter out = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),"UTF-8"));
		out.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		out.println();
		out.println("<VARD2>");
		out.println();
		out.println(getXMLTaggedText());
		out.println();
		out.println("</VARD2>");
		out.close();
		readFinished = true;
	}

	private String getXMLTaggedText() throws BadLocationException {
		String text = doc.getText(getStartOfDoc(), doc.getLength());

		highestPosToRead = words.size();
		currentPosInDoc = 0;
		TreeSet<Instance> instances = new TreeSet<Instance>();
		for(WordHolder wh : words) {
			instances.addAll(wh.getInstances());
			currentPosInDoc++;
		}

		int offset = 0;
		int prevTextLength = 0;
		
		highestPosToRead = instances.size();
		currentPosInDoc = 0;
		for(Instance instance : instances) {
			prevTextLength = text.length();
			text = instance.insertXMLTags(offset, text);
			offset += (text.length() - prevTextLength);
			currentPosInDoc++;
		}

		text = text.replaceAll("&", "&amp;");
		//text = text.replaceAll("\'", "&apos;");
				
		return text;		
	}



	public AttributeSet getAttributesAtPos(int pos) {
		return doc.getCharacterElement(pos).getAttributes();
	}

	public boolean isCharAtPosLetter(int pos) throws BadLocationException {
		return Character.isLetter(doc.getText(pos, 1).charAt(0));
	}

	public int getStartOfDoc() {
		return startPos.getOffset();
	}

	public int getEndOfDoc() {
		return endPos.getOffset();
	}

	public void findReplacementsForWord(WordHolder word) { //used if thread hasn't found words, ie does straight away
		//if(word.areReplacementsSet())
			//return;
		word.setReplacements(lud.findReplacements(word.getWord(), word.getSoundexCode(), confidenceWeights));
	}

	public int getReplacements() {
		readFinished = false;
		currentPosInDoc = 0;
		new GetRepsThread().start();
		highestPosToRead = words.size();
		return highestPosToRead;
	}
	
	private class GetRepsThread extends Thread {
		private GetRepsThread() {
			setPriority(4);
		}
		public void run() {
			currentPosInDoc = 0;
			for(WordHolder wh : words) {
				if(wh.isAtleastOneInstanceAVariant()) {
					findReplacementsForWord(wh);
				}
				currentPosInDoc++;
			}			
			readFinished = true;
		}
	}

	
	private class XMLParser implements ContentHandler {
		
		private static final String PARSER_NAME = "org.apache.xerces.parsers.SAXParser";
		
		private boolean printExtraXML, updateConfidenceWeights;
		private int textPos;
		
		private String freeText = "";
		private int freeTextStart = 0;
		private String currentWord = "";
		private int currentWordStart = 0;
		private boolean inJoin = false;
		String joinString;
		private boolean inFreeText = true;
		
		private ReplacementTempClass replacementInfo;
		
		private Vector<FutureInstance> variants = new Vector<FutureInstance>(200,100);
		private Vector<FutureReplaced> replaced = new Vector<FutureReplaced>(200,100);
		private Vector<FutureInstance> correct = new Vector<FutureInstance>(500,200);
		
		public XMLParser(boolean printExtraXML, boolean updateConfidenceWeights, String file) {
			this.printExtraXML = printExtraXML;
			this.updateConfidenceWeights = updateConfidenceWeights;
			this.textPos = 0;
			try {
				XMLReader parser = (XMLReader)Class.forName(PARSER_NAME).newInstance();
				parser.setContentHandler(this);
				parser.parse(file);
			} catch(IOException ex) {
				global.exceptionHandler.showException("Error reading file " + file, ex);
			} catch (InstantiationException ex) {
				global.exceptionHandler.showException("Error initiating XML Parser", ex);
			} catch (IllegalAccessException ex) {
				global.exceptionHandler.showException("Error initiating XML Parser", ex);
			} catch (ClassNotFoundException ex) {
				global.exceptionHandler.showException("Error initiating XML Parser", ex);
			} catch (SAXException ex) {
				global.exceptionHandler.showException("Error reading XML", ex);
			}

		}

		public void characters(char[] chars, int start, int end) throws SAXException {
			for(int i=start; i<end; i++) {
				try {
					doc.insertString(textPos++, "" + chars[i], null);
				} catch(BadLocationException e) {
					global.exceptionHandler.showException("Error in writing to text position - " + textPos, e);
				}
				
				if(inFreeText) {
					if(freeText.equals(""))
						freeTextStart = textPos-1;
					freeText += chars[i];
				}
				else {
					if(currentWord.equals(""))
						currentWordStart = textPos-1;
					currentWord += chars[i];
				}
			}
		}

		public void endDocument() throws SAXException {
			try {
				
				progressBarMax = variants.size() + replaced.size() + correct.size();
				
				for(FutureInstance v : variants) {
					String currentWordStringLC = v.word.toLowerCase();
					WordHolder currentWH = getReferenceTo(currentWordStringLC);
					VariantInstance instance = currentWH.addVariant(doc.createPosition(v.start), doc.createPosition(v.end), v.word);
					instance.setJoin(v.join);
					instance.setJoinString(v.joinString);
				}
				
				
			} catch(BadLocationException ex) {
				global.exceptionHandler.showException("Error occurred evaluating variant", ex);
			}
			
			try {
				for(FutureReplaced r : replaced) {
					xmlReplaceWord(doc.createPosition(r.start), doc.createPosition(r.end), r.word, r.repInfo, updateConfidenceWeights, r.join, r.joinString);
				}
			} catch(BadLocationException ex) {
				global.exceptionHandler.showException("Error evaluation replaced tag.", ex);
			} catch(BadXMLSyntaxException ex) {
				global.exceptionHandler.showException("Error replaced tag xml syntax", ex);					
			} catch(InvalidInstanceChangeException ex) {
				global.exceptionHandler.showException("Error occurred replacing instance", ex);
			}
			
			try {
				for(FutureInstance c : correct) {
					String currentWordStringLC = c.word.toLowerCase();
					WordHolder currentWH = getReferenceTo(currentWordStringLC);
					Instance instance;
					if(currentWH.isLowFreq())
						instance = currentWH.addRealError(doc.createPosition(c.start), doc.createPosition(c.end), c.word, RealErrorInstance.LOW_FREQ);
					else 
						instance = currentWH.addCorrect(doc.createPosition(c.start), doc.createPosition(c.end), c.word);
				
					instance.setJoin(c.join);
					instance.setJoinString(c.joinString);
				}
				
			} catch(BadLocationException ex) {
				global.exceptionHandler.showException("Error occurred evaluating joined correct word.", ex);
			}
			
			readFinished = true;
		}
		
		class FutureInstance {
			int start, end;
			String word;
			boolean join;
			String joinString;
			/**
			 * @param start
			 * @param end
			 * @param word
			 * @param join
			 * @param joinString
			 */
			FutureInstance(int start, int end, String word, boolean join, String joinString) {
				this.start = start;
				this.end = end;
				this.word = word;
				this.join = join;
				this.joinString = joinString;
			}
		}
		
		class FutureReplaced {
			ReplacementTempClass repInfo;
			int start, end;
			String word;
			boolean join;
			String joinString;
			/**
			 * @param repInfo
			 * @param start
			 * @param end
			 * @param word
			 * @param join
			 * @param joinString
			 */
			FutureReplaced(ReplacementTempClass repInfo, int start, int end, String word, boolean join, String joinString) {
				this.repInfo = repInfo;
				this.start = start;
				this.end = end;
				this.word = word;
				this.join = join;
				this.joinString = joinString;
			}
		}
		
		public void endElement(String namespaceURI, String localName, String qName) throws SAXException {
			if(inFreeText) {
				try {
					processFreeText();
				} catch(BadLocationException ex) {
					global.exceptionHandler.showException("Error reading free text.", ex);
				}
			}
			
			inFreeText = true;
			
			if(localName.equals("variant")) {
				
				variants.add(new FutureInstance(currentWordStart, currentWordStart+currentWord.length(), currentWord, inJoin, joinString));				
			
				currentWord = "";
				inJoin = false;
				joinString = null;
			}
			
			else if(localName.equals("replaced")) {
				
				replaced.add(new FutureReplaced(replacementInfo, currentWordStart, currentWordStart+currentWord.length(), currentWord, inJoin, joinString));
				
				currentWord = "";
				inJoin = false;
				joinString = null;
			}
			
			else if(localName.equals("join")) {
				if(!currentWord.equals("")) {
					
					correct.add(new FutureInstance(currentWordStart, currentWordStart+currentWord.length(), currentWord, true, joinString));
					
					inJoin = false;
					joinString = null;
				}
			}
			
			else if(localName.equals("VARD2")) {
				//ignore
			}
			
			else if(printExtraXML) {
				String toPrint = "</";
				toPrint += localName;
				toPrint += ">";
				
				try {
					doc.insertString(textPos++, toPrint, null);
					textPos+=(toPrint.length()-1);
				} catch(BadLocationException ex) {
					global.exceptionHandler.showException("Error writing extra xml.", ex);
				}
			}
			
		}

		public void endPrefixMapping(String arg0) throws SAXException {
			//do nothing
			
		}

		public void ignorableWhitespace(char[] chars, int start, int end) throws SAXException {
			for(int i=start; i<end; i++) {
				try {
					doc.insertString(textPos++, "" + chars[i], null);
				} catch(BadLocationException e) {
					global.exceptionHandler.showException("Error in writing to text position - " + textPos, e);
				}
				
				if(inFreeText) {
					if(freeText.equals(""))
						freeTextStart = textPos;
					freeText += chars[i];
				}
				else {
					if(currentWord.equals(""))
						currentWordStart = textPos;
					currentWord += chars[i];
				}
			}
			
		}

		public void processingInstruction(String arg0, String arg1) throws SAXException {
			//do nothing
			
		}

		public void setDocumentLocator(Locator arg0) {
			//do nothing
			
		}

		public void skippedEntity(String arg0) throws SAXException {
			//do nothing
			
		}

		public void startDocument() throws SAXException {
			readFinished = false;
			
		}

		public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {
			if(inFreeText) {
				try {
					processFreeText();
				} catch(BadLocationException ex) {
					global.exceptionHandler.showException("Error reading free text.", ex);
				}
			}
			
			if(localName.equals("variant")) {
				inFreeText = false;
				currentWord = "";
			}
			
			else if(localName.equals("replaced")) {
				replacementInfo =  new ReplacementTempClass();
				replacementInfo.variant = atts.getValue("", "variant");
				try {
					replacementInfo.ed = Integer.parseInt(atts.getValue("", "ed"));
				} catch(NumberFormatException ex) {
					global.exceptionHandler.showException("Edit distance attribute not a number.", ex);
				}
				
				String foundBy = atts.getValue("", "foundBy");
				String repType = atts.getValue("", "replacementType");
				
				if(foundBy.contains("s"))
					replacementInfo.soundex = true;
				
				if(foundBy.contains("l"))
					replacementInfo.lr = true;
				
				if(foundBy.contains("p"))
					replacementInfo.knownpre = true;
				
				else if(foundBy.contains("u"))
					replacementInfo.knownua = true;
				
				if(repType.contains("u"))
					replacementInfo.uaToDic = true;
				
				if(repType.contains("r"))
					replacementInfo.inDict = true;
				
				inFreeText = false;
				currentWord = "";
			}
			
			else if(localName.equals("join")) {
				inJoin = true;
				joinString = atts.getValue("", "original");
				inFreeText = false;
				currentWord = "";
			}
			
			else if(localName.equals("VARD2")) {
				//ignore
			}
			
			else if(printExtraXML) {
				String toPrint = "<";
				toPrint += localName;
				for(int i=0;i<atts.getLength();i++) {
					toPrint += " " + atts.getLocalName(i) + "=\"" + atts.getValue(i) + "\"";
				}
				toPrint += ">";
				
				try {
					doc.insertString(textPos++, toPrint, null);
					textPos+=(toPrint.length()-1);
				} catch(BadLocationException ex) {
					global.exceptionHandler.showException("Error writing extra xml.", ex);
				}
			}
		}
		
		public void processFreeText() throws BadLocationException {
			if(freeText.equals(""))
				return;
			
			StringCharacterIterator sci = new StringCharacterIterator(freeText);
			char currentLetter = sci.first();
			char lastLetter;
			int currentPos = sci.getIndex();
			int currentStart = 0;
			int currentEnd = 0;
			currentPosInDoc = 0;
			String currentWordString;
			boolean inWord;
			if(isLetter(currentLetter))
				inWord = true;
			else
				inWord = false;
			do {
				lastLetter = currentLetter;
				currentLetter = sci.next();
				currentPos = sci.getIndex();

				if(inWord) {
					if(!isLetter(currentLetter) && !(currentLetter == '-')) {
						if(lastLetter == '-')
							currentEnd = currentPos-1;
						else
							currentEnd = currentPos;

						if((currentEnd-currentStart)>0) {
							currentWordString = freeText.substring(currentStart,currentEnd);
							if(isValidWord(currentWordString)) {
								correct.add(new FutureInstance(freeTextStart+currentStart, freeTextStart+currentStart+currentWordString.length(), currentWordString, false, null));
							}
						}
								
						inWord = false;
					}
				}
				else if(isLetter(currentLetter)) {
					currentStart = currentPos;
					inWord = true;
				}
			} while(currentLetter != StringCharacterIterator.DONE);
			
			freeText = "";			
		}

		public void startPrefixMapping(String arg0, String arg1) throws SAXException {
			//do nothing
			
		}

	}
	
	public String[] getShortCounts() {
		return this.shortCounts;
	}
	
	//TreeSet<SuggestedReplacement> repsFound = new TreeSet<SuggestedReplacement>();
	
	private void xmlReplaceWord(Position start, Position end, String replacement, ReplacementTempClass rtc, boolean updateConfidenceWeights, boolean inJoin, String joinString) throws BadLocationException, BadXMLSyntaxException, InvalidInstanceChangeException {
		
		Word wordInDic;
		
		if(!rtc.inDict) {
			wordInDic = new Word(replacement, null, rtc.uaToDic, false, 1);
		}
		else {
			wordInDic = lud.checkWords(replacement);
			if(wordInDic == null) {
				if(rtc.uaToDic)
					wordInDic = lud.addUserWord(replacement);
				else
					throw new BadXMLSyntaxException("Replacement stated as pre-defined dictionary word incorrectly");
			}
		}
		
		SuggestedReplacement rep = new SuggestedReplacement(wordInDic, rtc.variant, confidenceWeights);
		rep.setEditDistance(rtc.ed);
		rep.setKnownVariant((rtc.knownua || rtc.knownpre), rtc.knownua);
		rep.setLetterReplacement(rtc.lr);
		rep.setSoundex(rtc.soundex);
		
		/*if(!repsFound.add(rep)) {
			Iterator<SuggestedReplacement> it = repsFound.iterator();
			while(it.hasNext()) {
				SuggestedReplacement current = it.next();
				if(current.equals(rep)) {
					rep = current;
					break;
				}
			}
		}
		*/

		
		WordHolder wh = getReferenceTo(rtc.variant.toLowerCase());
		
		VariantInstance vi = wh.addVariant(start, end, rtc.variant);
		WordHolder.ReplaceFInstanceEdit<VariantInstance> edit = new WordHolder.ReplaceFInstanceEdit<VariantInstance>(vi, rep, false);
		edit.execute();
		
		ReplacedInstance ri = edit.getReplacedInstance();
		
		ri.setJoin(inJoin);
		ri.setJoinString(joinString);
	}

	
	public int getCurrentPosInDoc() {
		return this.currentPosInDoc;
	}

	public void setCurrentPosInDoc(int currentPosInDoc) {
		this.currentPosInDoc = currentPosInDoc;
	}

	public int getHighestPosToRead() {
		return this.highestPosToRead;
	}

	public void setHighestPosToRead(int highestPosToRead) {
		this.highestPosToRead = highestPosToRead;
	}

	public LookUpDictionary getLud() {
		return lud;
	}

	public void setLud(LookUpDictionary l) {
		lud = l;
	}

	public void setReadFinished(boolean readFinished) {
		this.readFinished = readFinished;
	}
	
	public class ReplacementTempClass {
		public boolean lr = false;
		public boolean soundex = false;
		public boolean knownpre = false;
		public boolean knownua = false;
		public boolean inDict = false;
		public boolean uaToDic = false;
		public String variant;
		public int ed;
	}

	public String getStats() {
		int wordsCount = words.size();
		int variantsCount = 0, replacedCount = 0, correctCount = 0, realErrorCount = 0;
		
		for(WordHolder wh : words) {
			if(wh.isAtleastOneInstanceAVariant()) {
				variantsCount++;
			}
			if(wh.isAtleastOneInstanceAReplaced()) {
				replacedCount++;
			}
			if(wh.isAtleastOneInstanceACorrect()) {
				correctCount++;
			}
			if(wh.isAtleastOneInstanceARealError()) {
				realErrorCount++;
			}
		}
		
		return wordsCount + "\t" + variantsCount + "\t" + replacedCount + "\t" + correctCount + "\t" + realErrorCount;
	}
}